

/** \file     TComRom.cpp
    \brief    global variables & functions
*/

#include "TComRom.h"
#include <memory.h>
#include <stdlib.h>
#include <stdio.h>

// ====================================================================================================================
// Initialize / destroy functions
// ====================================================================================================================

// initialize ROM variables
Void initROM()
{
  Int i, c;
  
  // g_aucConvertToBit[ x ]: log2(x/4), if x=4 -> 0, x=8 -> 1, x=16 -> 2, ...
  ::memset( g_aucConvertToBit,   -1, sizeof( g_aucConvertToBit ) );
  c=0;
  for ( i=4; i<MAX_CU_SIZE; i*=2 )
  {
    g_aucConvertToBit[ i ] = c;
    c++;
  }
  g_aucConvertToBit[ i ] = c;
  
  // g_auiFrameScanXY[ g_aucConvertToBit[ transformSize ] ]: zigzag scan array for transformSize
  c=2;
  for ( i=0; i<MAX_CU_DEPTH; i++ )
  {
    g_auiFrameScanXY[ i ] = new UInt[ c*c ];
    g_auiFrameScanX [ i ] = new UInt[ c*c ];
    g_auiFrameScanY [ i ] = new UInt[ c*c ];
    initFrameScanXY( g_auiFrameScanXY[i], g_auiFrameScanX[i], g_auiFrameScanY[i], c, c );
#if QC_MDCS
    g_auiSigLastScan[0][i] = new UInt[ c*c ];
    g_auiSigLastScan[1][i] = new UInt[ c*c ];
    g_auiSigLastScan[2][i] = new UInt[ c*c ];
    initSigLastScan( g_auiSigLastScan[0][i], g_auiSigLastScan[1][i], g_auiSigLastScan[2][i], c, c, i);
#endif //QC_MDCS

    c <<= 1;
  }  
}

Void destroyROM()
{
  Int i;
  
  for ( i=0; i<MAX_CU_DEPTH; i++ )
  {
    delete[] g_auiFrameScanXY[i];
    delete[] g_auiFrameScanX [i];
    delete[] g_auiFrameScanY [i];
#if QC_MDCS
    delete[] g_auiSigLastScan[0][i];
    delete[] g_auiSigLastScan[1][i];
    delete[] g_auiSigLastScan[2][i];
#endif //QC_MDCS
  }

#if HHI_DMM_INTRA
  if ( !g_aacWedgeLists.empty() )
  {
    for ( UInt ui = 0; ui < g_aacWedgeLists.size(); ui++ )
    {
      g_aacWedgeLists[ui].clear();
    }
    g_aacWedgeLists.clear();
  }

  if ( !g_aacWedgeRefLists.empty() )
  {
    for ( UInt ui = 0; ui < g_aacWedgeRefLists.size(); ui++ )
    {
      g_aacWedgeRefLists[ui].clear();
    }
    g_aacWedgeRefLists.clear();
  }
#endif
}

// ====================================================================================================================
// Data structure related table & variable
// ====================================================================================================================

UInt g_uiMaxCUWidth  = MAX_CU_SIZE;
UInt g_uiMaxCUHeight = MAX_CU_SIZE;
UInt g_uiMaxCUDepth  = MAX_CU_DEPTH;
UInt g_uiAddCUDepth  = 0;

UInt g_auiZscanToRaster [ MAX_NUM_SPU_W*MAX_NUM_SPU_W ] = { 0, };
UInt g_auiRasterToZscan [ MAX_NUM_SPU_W*MAX_NUM_SPU_W ] = { 0, };
UInt g_auiRasterToPelX  [ MAX_NUM_SPU_W*MAX_NUM_SPU_W ] = { 0, };
UInt g_auiRasterToPelY  [ MAX_NUM_SPU_W*MAX_NUM_SPU_W ] = { 0, };

UInt g_auiPUOffset[4] = { 0, 8, 4, 4 };

Void initZscanToRaster ( Int iMaxDepth, Int iDepth, UInt uiStartVal, UInt*& rpuiCurrIdx )
{
  Int iStride = 1 << ( iMaxDepth - 1 );
  
  if ( iDepth == iMaxDepth )
  {
    rpuiCurrIdx[0] = uiStartVal;
    rpuiCurrIdx++;
  }
  else
  {
    Int iStep = iStride >> iDepth;
    initZscanToRaster( iMaxDepth, iDepth+1, uiStartVal,                     rpuiCurrIdx );
    initZscanToRaster( iMaxDepth, iDepth+1, uiStartVal+iStep,               rpuiCurrIdx );
    initZscanToRaster( iMaxDepth, iDepth+1, uiStartVal+iStep*iStride,       rpuiCurrIdx );
    initZscanToRaster( iMaxDepth, iDepth+1, uiStartVal+iStep*iStride+iStep, rpuiCurrIdx );
  }
}

Void initRasterToZscan ( UInt uiMaxCUWidth, UInt uiMaxCUHeight, UInt uiMaxDepth )
{
  UInt  uiMinCUWidth  = uiMaxCUWidth  >> ( uiMaxDepth - 1 );
  UInt  uiMinCUHeight = uiMaxCUHeight >> ( uiMaxDepth - 1 );
  
  UInt  uiNumPartInWidth  = (UInt)uiMaxCUWidth  / uiMinCUWidth;
  UInt  uiNumPartInHeight = (UInt)uiMaxCUHeight / uiMinCUHeight;
  
  for ( UInt i = 0; i < uiNumPartInWidth*uiNumPartInHeight; i++ )
  {
    g_auiRasterToZscan[ g_auiZscanToRaster[i] ] = i;
  }
}

Void initRasterToPelXY ( UInt uiMaxCUWidth, UInt uiMaxCUHeight, UInt uiMaxDepth )
{
  UInt    i;
  
  UInt* uiTempX = &g_auiRasterToPelX[0];
  UInt* uiTempY = &g_auiRasterToPelY[0];
  
  UInt  uiMinCUWidth  = uiMaxCUWidth  >> ( uiMaxDepth - 1 );
  UInt  uiMinCUHeight = uiMaxCUHeight >> ( uiMaxDepth - 1 );
  
  UInt  uiNumPartInWidth  = uiMaxCUWidth  / uiMinCUWidth;
  UInt  uiNumPartInHeight = uiMaxCUHeight / uiMinCUHeight;
  
  uiTempX[0] = 0; uiTempX++;
  for ( i = 1; i < uiNumPartInWidth; i++ )
  {
    uiTempX[0] = uiTempX[-1] + uiMinCUWidth; uiTempX++;
  }
  for ( i = 1; i < uiNumPartInHeight; i++ )
  {
    memcpy(uiTempX, uiTempX-uiNumPartInWidth, sizeof(UInt)*uiNumPartInWidth);
    uiTempX += uiNumPartInWidth;
  }
  
  for ( i = 1; i < uiNumPartInWidth*uiNumPartInHeight; i++ )
  {
    uiTempY[i] = ( i / uiNumPartInWidth ) * uiMinCUWidth;
  }
};

#if MTK_SAO
Int  LevelRowCol2Idx(int level, int row, int col)
{
  Int idx;
  if (level == 0)
  {
    idx = 0;
  }
  else if (level == 1)
  {
    idx = 1 + row*2 + col;
  }
  else if (level == 2)
  {
    idx = 5 + row*4 + col;
  }
  else if (level == 3)
  {
    idx = 21 + row*8 + col;
  }
  else // (level == 4)
  {
    idx = 85 + row*16 + col;
  }
  return idx;
}

void Idx2LevelRowCol(int idx, int *level, int *row, int *col)
{
  if (idx == 0)
  {
    *level = 0;
    *row = 0;
    *col = 0;
  }
  else if (idx>=1 && idx<=4)
  {
    *level = 1;
    *row = (idx-1) / 2;
    *col = (idx-1) % 2;
  }
  else if (idx>=5 && idx<=20)
  {
    *level = 2;
    *row = (idx-5) / 4;
    *col = (idx-5) % 4;
  }
  else if (idx>=21 && idx<=84)
  {
    *level = 3;
    *row = (idx-21) / 8;
    *col = (idx-21) % 8;
  }
  else // (idx>=85 && idx<=340)
  {
    *level = 4;
    *row = (idx-85) / 16;
    *col = (idx-85) % 16;
  }
}
#endif

#if E243_CORE_TRANSFORMS
  UInt g_auiQ[6] = {26214,23302,20560,18396,16384,14564};    
  UInt g_auiIQ[6] = {40,45,51,57,64,72};

  const short g_aiT4[4][4] = {
    { 64, 64, 64, 64},
    { 83, 36,-36,-83},
    { 64,-64,-64, 64},
    { 36,-83, 83,-36}
  };

  const short g_aiT8[8][8] = {
    { 64, 64, 64, 64, 64, 64, 64, 64},
    { 89, 75, 50, 18,-18,-50,-75,-89},
    { 83, 36,-36,-83,-83,-36, 36, 83},
    { 75,-18,-89,-50, 50, 89, 18,-75},
    { 64,-64,-64, 64, 64,-64,-64, 64},
    { 50,-89, 18, 75,-75,-18, 89,-50},
    { 36,-83, 83,-36,-36, 83,-83, 36},
    { 18,-50, 75,-89, 89,-75, 50,-18}
  };

 const short g_aiT16[16][16] = {
    { 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64},
    { 90, 87, 80, 70, 57, 43, 25,  9, -9,-25,-43,-57,-70,-80,-87,-90},
    { 89, 75, 50, 18,-18,-50,-75,-89,-89,-75,-50,-18, 18, 50, 75, 89},
    { 87, 57,  9,-43,-80,-90,-70,-25, 25, 70, 90, 80, 43, -9,-57,-87},
    { 83, 36,-36,-83,-83,-36, 36, 83, 83, 36,-36,-83,-83,-36, 36, 83},
    { 80,  9,-70,-87,-25, 57, 90, 43,-43,-90,-57, 25, 87, 70, -9,-80},
    { 75,-18,-89,-50, 50, 89, 18,-75,-75, 18, 89, 50,-50,-89,-18, 75},
    { 70,-43,-87,  9, 90, 25,-80,-57, 57, 80,-25,-90, -9, 87, 43,-70},
    { 64,-64,-64, 64, 64,-64,-64, 64, 64,-64,-64, 64, 64,-64,-64, 64},
    { 57,-80,-25, 90, -9,-87, 43, 70,-70,-43, 87,  9,-90, 25, 80,-57},
    { 50,-89, 18, 75,-75,-18, 89,-50,-50, 89,-18,-75, 75, 18,-89, 50},
    { 43,-90, 57, 25,-87, 70,  9,-80, 80, -9,-70, 87,-25,-57, 90,-43},
    { 36,-83, 83,-36,-36, 83,-83, 36, 36,-83, 83,-36,-36, 83,-83, 36},
    { 25,-70, 90,-80, 43,  9,-57, 87,-87, 57, -9,-43, 80,-90, 70,-25},
    { 18,-50, 75,-89, 89,-75, 50,-18,-18, 50,-75, 89,-89, 75,-50, 18},
    {  9,-25, 43,-57, 70,-80, 87,-90, 90,-87, 80,-70, 57,-43, 25, -9}
 };

  const short g_aiT32[32][32] = {
    { 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64},
    { 90, 90, 88, 85, 82, 78, 73, 67, 61, 54, 46, 38, 31, 22, 13,  4, -4,-13,-22,-31,-38,-46,-54,-61,-67,-73,-78,-82,-85,-88,-90,-90},
    { 90, 87, 80, 70, 57, 43, 25,  9, -9,-25,-43,-57,-70,-80,-87,-90,-90,-87,-80,-70,-57,-43,-25, -9,  9, 25, 43, 57, 70, 80, 87, 90},
    { 90, 82, 67, 46, 22, -4,-31,-54,-73,-85,-90,-88,-78,-61,-38,-13, 13, 38, 61, 78, 88, 90, 85, 73, 54, 31,  4,-22,-46,-67,-82,-90},
    { 89, 75, 50, 18,-18,-50,-75,-89,-89,-75,-50,-18, 18, 50, 75, 89, 89, 75, 50, 18,-18,-50,-75,-89,-89,-75,-50,-18, 18, 50, 75, 89},
    { 88, 67, 31,-13,-54,-82,-90,-78,-46, -4, 38, 73, 90, 85, 61, 22,-22,-61,-85,-90,-73,-38,  4, 46, 78, 90, 82, 54, 13,-31,-67,-88},
    { 87, 57,  9,-43,-80,-90,-70,-25, 25, 70, 90, 80, 43, -9,-57,-87,-87,-57, -9, 43, 80, 90, 70, 25,-25,-70,-90,-80,-43,  9, 57, 87},
    { 85, 46,-13,-67,-90,-73,-22, 38, 82, 88, 54, -4,-61,-90,-78,-31, 31, 78, 90, 61,  4,-54,-88,-82,-38, 22, 73, 90, 67, 13,-46,-85},
    { 83, 36,-36,-83,-83,-36, 36, 83, 83, 36,-36,-83,-83,-36, 36, 83, 83, 36,-36,-83,-83,-36, 36, 83, 83, 36,-36,-83,-83,-36, 36, 83},
    { 82, 22,-54,-90,-61, 13, 78, 85, 31,-46,-90,-67,  4, 73, 88, 38,-38,-88,-73, -4, 67, 90, 46,-31,-85,-78,-13, 61, 90, 54,-22,-82},
    { 80,  9,-70,-87,-25, 57, 90, 43,-43,-90,-57, 25, 87, 70, -9,-80,-80, -9, 70, 87, 25,-57,-90,-43, 43, 90, 57,-25,-87,-70,  9, 80},
    { 78, -4,-82,-73, 13, 85, 67,-22,-88,-61, 31, 90, 54,-38,-90,-46, 46, 90, 38,-54,-90,-31, 61, 88, 22,-67,-85,-13, 73, 82,  4,-78},
    { 75,-18,-89,-50, 50, 89, 18,-75,-75, 18, 89, 50,-50,-89,-18, 75, 75,-18,-89,-50, 50, 89, 18,-75,-75, 18, 89, 50,-50,-89,-18, 75},
    { 73,-31,-90,-22, 78, 67,-38,-90,-13, 82, 61,-46,-88, -4, 85, 54,-54,-85,  4, 88, 46,-61,-82, 13, 90, 38,-67,-78, 22, 90, 31,-73},
    { 70,-43,-87,  9, 90, 25,-80,-57, 57, 80,-25,-90, -9, 87, 43,-70,-70, 43, 87, -9,-90,-25, 80, 57,-57,-80, 25, 90,  9,-87,-43, 70},
    { 67,-54,-78, 38, 85,-22,-90,  4, 90, 13,-88,-31, 82, 46,-73,-61, 61, 73,-46,-82, 31, 88,-13,-90, -4, 90, 22,-85,-38, 78, 54,-67},
    { 64,-64,-64, 64, 64,-64,-64, 64, 64,-64,-64, 64, 64,-64,-64, 64, 64,-64,-64, 64, 64,-64,-64, 64, 64,-64,-64, 64, 64,-64,-64, 64},
    { 61,-73,-46, 82, 31,-88,-13, 90, -4,-90, 22, 85,-38,-78, 54, 67,-67,-54, 78, 38,-85,-22, 90,  4,-90, 13, 88,-31,-82, 46, 73,-61},
    { 57,-80,-25, 90, -9,-87, 43, 70,-70,-43, 87,  9,-90, 25, 80,-57,-57, 80, 25,-90,  9, 87,-43,-70, 70, 43,-87, -9, 90,-25,-80, 57},
    { 54,-85, -4, 88,-46,-61, 82, 13,-90, 38, 67,-78,-22, 90,-31,-73, 73, 31,-90, 22, 78,-67,-38, 90,-13,-82, 61, 46,-88,  4, 85,-54},
    { 50,-89, 18, 75,-75,-18, 89,-50,-50, 89,-18,-75, 75, 18,-89, 50, 50,-89, 18, 75,-75,-18, 89,-50,-50, 89,-18,-75, 75, 18,-89, 50},
    { 46,-90, 38, 54,-90, 31, 61,-88, 22, 67,-85, 13, 73,-82,  4, 78,-78, -4, 82,-73,-13, 85,-67,-22, 88,-61,-31, 90,-54,-38, 90,-46},
    { 43,-90, 57, 25,-87, 70,  9,-80, 80, -9,-70, 87,-25,-57, 90,-43,-43, 90,-57,-25, 87,-70, -9, 80,-80,  9, 70,-87, 25, 57,-90, 43},
    { 38,-88, 73, -4,-67, 90,-46,-31, 85,-78, 13, 61,-90, 54, 22,-82, 82,-22,-54, 90,-61,-13, 78,-85, 31, 46,-90, 67,  4,-73, 88,-38},
    { 36,-83, 83,-36,-36, 83,-83, 36, 36,-83, 83,-36,-36, 83,-83, 36, 36,-83, 83,-36,-36, 83,-83, 36, 36,-83, 83,-36,-36, 83,-83, 36},
    { 31,-78, 90,-61,  4, 54,-88, 82,-38,-22, 73,-90, 67,-13,-46, 85,-85, 46, 13,-67, 90,-73, 22, 38,-82, 88,-54, -4, 61,-90, 78,-31},
    { 25,-70, 90,-80, 43,  9,-57, 87,-87, 57, -9,-43, 80,-90, 70,-25,-25, 70,-90, 80,-43, -9, 57,-87, 87,-57,  9, 43,-80, 90,-70, 25},
    { 22,-61, 85,-90, 73,-38, -4, 46,-78, 90,-82, 54,-13,-31, 67,-88, 88,-67, 31, 13,-54, 82,-90, 78,-46,  4, 38,-73, 90,-85, 61,-22},
    { 18,-50, 75,-89, 89,-75, 50,-18,-18, 50,-75, 89,-89, 75,-50, 18, 18,-50, 75,-89, 89,-75, 50,-18,-18, 50,-75, 89,-89, 75,-50, 18},
    { 13,-38, 61,-78, 88,-90, 85,-73, 54,-31,  4, 22,-46, 67,-82, 90,-90, 82,-67, 46,-22, -4, 31,-54, 73,-85, 90,-88, 78,-61, 38,-13},
    {  9,-25, 43,-57, 70,-80, 87,-90, 90,-87, 80,-70, 57,-43, 25, -9, -9, 25,-43, 57,-70, 80,-87, 90,-90, 87,-80, 70,-57, 43,-25,  9},
    {  4,-13, 22,-31, 38,-46, 54,-61, 67,-73, 78,-82, 85,-88, 90,-90, 90,-90, 88,-85, 82,-78, 73,-67, 61,-54, 46,-38, 31,-22, 13, -4}
  };

#else //E243_CORE_TRANSFORMS
// ====================================================================================================================
// Quantization & DeQuantization
// ====================================================================================================================

UInt g_aiQuantCoef4[6] =
{
  102, 93, 79, 73, 64, 57
};

Int g_aiDequantCoef4[6] =
{
  10, 11, 13, 14, 16,18
};

UInt g_aiQuantCoef[6][16] =
{
  {
    13107, 8066,13107, 8066,
    8066, 5243, 8066, 5243,
    13107, 8066,13107, 8066,
    8066, 5243, 8066, 5243
  },
  {
    11916, 7490,11916, 7490,
    7490, 4660, 7490, 4660,
    11916, 7490,11916, 7490,
    7490, 4660, 7490, 4660
  },
  {
    10082, 6554,10082, 6554,
    6554, 4194, 6554, 4194,
    10082, 6554,10082, 6554,
    6554, 4194, 6554, 4194
  },
  {
    9362, 5825, 9362, 5825,
    5825, 3647, 5825, 3647,
    9362, 5825, 9362, 5825,
    5825, 3647, 5825, 3647
  },
  {
    8192, 5243, 8192, 5243,
    5243, 3355, 5243, 3355,
    8192, 5243, 8192, 5243,
    5243, 3355, 5243, 3355
  },
  {
    7282, 4559, 7282, 4559,
    4559, 2893, 4559, 2893,
    7282, 4559, 7282, 4559,
    4559, 2893, 4559, 2893
  }
};

Int g_aiDequantCoef[6][16] =
{
  {
    10, 13, 10, 13,
    13, 16, 13, 16,
    10, 13, 10, 13,
    13, 16, 13, 16
  },
  {
    11, 14, 11, 14,
    14, 18, 14, 18,
    11, 14, 11, 14,
    14, 18, 14, 18
  },
  {
    13, 16, 13, 16,
    16, 20, 16, 20,
    13, 16, 13, 16,
    16, 20, 16, 20
  },
  {
    14, 18, 14, 18,
    18, 23, 18, 23,
    14, 18, 14, 18,
    18, 23, 18, 23
  },
  {
    16, 20, 16, 20,
    20, 25, 20, 25,
    16, 20, 16, 20,
    20, 25, 20, 25
  },
  {
    18, 23, 18, 23,
    23, 29, 23, 29,
    18, 23, 18, 23,
    23, 29, 23, 29
  }
};

UInt g_aiQuantCoef4096[6] =
{
  26, 23, 20, 18, 16, 14
}; // 9
// {51, 47, 39, 37, 32, 28}; // 10
// {102, 93, 79, 73, 64, 57}; // 11

UInt g_aiDeQuantCoef4096[6] =
{
  10, 11, 13, 14, 16, 18
}; // 9
// {20, 22, 26, 28, 32, 36}; // 10
// {40, 44, 52, 56, 64, 72}; // 11

UInt g_aiQuantCoef1024[6][1024] =
{
  {
    102, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 102, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103,
    103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 104, 103, 104, 103, 104, 103, 104, 103, 104, 103, 104, 103, 103, 103, 103, 104, 103, 103, 103, 103, 103,
    103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 104, 103, 104, 103, 104, 103, 104, 103, 104, 103, 104, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103,
    103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 104, 103, 104, 103, 104, 103, 104, 103, 104, 103, 104, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103,
    103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103,
    103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 104, 103, 104, 103, 104, 103, 104, 103, 104, 103, 104, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103,
    103, 103, 103, 103, 103, 103, 104, 103, 103, 103, 104, 104, 103, 104, 104, 104, 103, 104, 104, 104, 103, 104, 104, 103, 103, 103, 104, 103, 103, 103, 103, 103,
    103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103,
    103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103,
    103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 104, 103, 104, 103, 104, 103, 104, 103, 104, 103, 104, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103,
    103, 103, 103, 103, 103, 103, 104, 103, 103, 103, 104, 104, 103, 104, 104, 104, 103, 104, 104, 104, 103, 104, 104, 103, 103, 103, 104, 103, 103, 103, 103, 103,
    103, 104, 104, 104, 103, 104, 104, 103, 103, 104, 104, 104, 103, 104, 104, 104, 103, 104, 104, 104, 103, 104, 104, 104, 103, 103, 104, 104, 103, 104, 104, 104,
    103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103,
    103, 104, 104, 104, 103, 104, 104, 103, 103, 104, 104, 104, 103, 104, 104, 104, 103, 104, 104, 104, 103, 104, 104, 104, 103, 103, 104, 104, 103, 104, 104, 104,
    103, 103, 103, 103, 103, 103, 104, 103, 103, 103, 104, 104, 103, 104, 104, 104, 103, 104, 104, 104, 103, 104, 104, 103, 103, 103, 104, 103, 103, 103, 103, 103,
    103, 104, 104, 104, 103, 104, 104, 103, 103, 104, 104, 104, 103, 104, 104, 104, 103, 104, 104, 104, 103, 104, 104, 104, 103, 103, 104, 104, 103, 104, 104, 104,
    102, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 102, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103,
    103, 104, 104, 104, 103, 104, 104, 103, 103, 104, 104, 104, 103, 104, 104, 104, 103, 104, 104, 104, 103, 104, 104, 104, 103, 103, 104, 104, 103, 104, 104, 104,
    103, 103, 103, 103, 103, 103, 104, 103, 103, 103, 104, 104, 103, 104, 104, 104, 103, 104, 104, 104, 103, 104, 104, 103, 103, 103, 104, 103, 103, 103, 103, 103,
    103, 104, 104, 104, 103, 104, 104, 103, 103, 104, 104, 104, 103, 104, 104, 104, 103, 104, 104, 104, 103, 104, 104, 104, 103, 103, 104, 104, 103, 104, 104, 104,
    103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103,
    103, 104, 104, 104, 103, 104, 104, 103, 103, 104, 104, 104, 103, 104, 104, 104, 103, 104, 104, 104, 103, 104, 104, 104, 103, 103, 104, 104, 103, 104, 104, 104,
    103, 103, 103, 103, 103, 103, 104, 103, 103, 103, 104, 104, 103, 104, 104, 104, 103, 104, 104, 104, 103, 104, 104, 103, 103, 103, 104, 103, 103, 103, 103, 103,
    103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 104, 103, 104, 103, 104, 103, 104, 103, 104, 103, 104, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103,
    103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103,
    103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103,
    103, 104, 103, 103, 103, 103, 104, 103, 103, 103, 104, 104, 103, 104, 104, 104, 103, 104, 104, 104, 103, 104, 104, 103, 103, 103, 104, 103, 103, 103, 103, 103,
    103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 104, 103, 104, 103, 104, 103, 104, 103, 104, 103, 104, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103,
    103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103,
    103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 104, 103, 104, 103, 104, 103, 104, 103, 104, 103, 104, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103,
    103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 104, 103, 104, 103, 104, 103, 104, 103, 104, 103, 104, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103,
    103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 104, 103, 104, 103, 104, 103, 104, 103, 104, 103, 104, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103
  },
  {
    93, 94, 94, 94, 93, 94, 94, 93, 93, 94, 94, 94, 93, 94, 94, 94, 93, 94, 94, 94, 93, 94, 94, 94, 93, 93, 94, 94, 93, 94, 94, 94,
    94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94,
    94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94,
    94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94,
    93, 94, 94, 94, 93, 94, 94, 93, 93, 94, 94, 94, 93, 94, 94, 94, 93, 94, 94, 94, 93, 94, 94, 94, 93, 93, 94, 94, 93, 94, 94, 94,
    94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94,
    94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94,
    93, 94, 94, 94, 93, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 93, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 93, 94, 94, 94,
    93, 94, 94, 94, 93, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 93, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 93, 94, 94, 94,
    94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94,
    94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94,
    94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94,
    93, 94, 94, 94, 93, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 93, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 93, 94, 94, 94,
    94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94,
    94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94,
    94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94,
    93, 94, 94, 94, 93, 94, 94, 93, 93, 94, 94, 94, 93, 94, 94, 94, 93, 94, 94, 94, 93, 94, 94, 94, 93, 93, 94, 94, 93, 94, 94, 94,
    94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94,
    94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94,
    94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94,
    93, 94, 94, 94, 93, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 93, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 93, 94, 94, 94,
    94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94,
    94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94,
    94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94,
    93, 94, 94, 94, 93, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 93, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94,
    93, 94, 94, 94, 93, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 93, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 93, 94, 94, 94,
    94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94,
    94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94,
    93, 94, 94, 94, 93, 94, 94, 93, 93, 94, 94, 94, 93, 94, 94, 94, 93, 94, 94, 94, 93, 94, 94, 94, 94, 93, 94, 94, 93, 94, 94, 94,
    94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94,
    94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94,
    94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94
  },
  {
    79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79,
    79, 80, 80, 80, 79, 80, 80, 79, 79, 80, 80, 80, 79, 80, 80, 80, 79, 80, 80, 80, 79, 80, 80, 80, 79, 79, 80, 80, 79, 80, 80, 80,
    79, 80, 80, 80, 79, 80, 80, 79, 79, 79, 80, 80, 79, 80, 80, 80, 79, 80, 80, 80, 79, 80, 80, 79, 79, 79, 80, 80, 79, 80, 80, 80,
    79, 80, 80, 79, 79, 80, 80, 79, 79, 79, 80, 80, 79, 80, 80, 80, 79, 80, 80, 80, 79, 80, 80, 79, 79, 79, 80, 80, 79, 79, 80, 80,
    79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79,
    79, 80, 80, 80, 79, 80, 80, 79, 79, 80, 80, 80, 79, 80, 80, 80, 79, 80, 80, 80, 79, 80, 80, 80, 79, 79, 80, 80, 79, 80, 80, 80,
    79, 80, 80, 80, 79, 80, 80, 79, 79, 80, 80, 80, 79, 80, 80, 80, 79, 80, 80, 80, 79, 80, 80, 80, 79, 79, 80, 80, 79, 80, 80, 80,
    79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 80, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79,
    79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 80, 79, 80, 79, 80, 79, 80, 79, 80, 79, 80, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79,
    79, 80, 79, 79, 79, 80, 80, 79, 79, 79, 80, 80, 79, 80, 80, 80, 79, 80, 80, 80, 79, 80, 80, 79, 79, 79, 80, 79, 79, 79, 79, 80,
    79, 80, 80, 80, 79, 80, 80, 79, 79, 80, 80, 80, 79, 80, 80, 80, 79, 80, 80, 80, 79, 80, 80, 80, 79, 79, 80, 80, 79, 80, 80, 80,
    79, 80, 80, 80, 79, 80, 80, 79, 80, 80, 80, 80, 80, 80, 80, 80, 79, 80, 80, 80, 80, 80, 80, 80, 80, 80, 80, 80, 79, 80, 80, 80,
    79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 80, 79, 80, 79, 80, 79, 80, 79, 80, 79, 80, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79,
    79, 80, 80, 80, 79, 80, 80, 79, 80, 80, 80, 80, 80, 80, 80, 80, 79, 80, 80, 80, 80, 80, 80, 80, 80, 79, 80, 80, 79, 80, 80, 80,
    79, 80, 80, 80, 79, 80, 80, 79, 79, 80, 80, 80, 79, 80, 80, 80, 79, 80, 80, 80, 79, 80, 80, 80, 79, 79, 80, 80, 79, 80, 80, 80,
    79, 80, 80, 80, 79, 80, 80, 79, 80, 80, 80, 80, 80, 80, 80, 80, 79, 80, 80, 80, 80, 80, 80, 80, 80, 80, 80, 80, 79, 80, 80, 80,
    79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79,
    79, 80, 80, 80, 79, 80, 80, 80, 80, 80, 80, 80, 80, 80, 80, 80, 79, 80, 80, 80, 80, 80, 80, 80, 80, 80, 80, 80, 79, 80, 80, 80,
    79, 80, 80, 80, 79, 80, 80, 79, 79, 80, 80, 80, 79, 80, 80, 80, 79, 80, 80, 80, 79, 80, 80, 80, 79, 79, 80, 80, 79, 80, 80, 80,
    79, 80, 80, 80, 79, 80, 80, 79, 80, 80, 80, 80, 80, 80, 80, 80, 79, 80, 80, 80, 80, 80, 80, 80, 80, 79, 80, 80, 79, 80, 80, 80,
    79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 80, 79, 80, 79, 80, 79, 80, 79, 80, 79, 80, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79,
    79, 80, 80, 80, 79, 80, 80, 79, 80, 80, 80, 80, 80, 80, 80, 80, 79, 80, 80, 80, 80, 80, 80, 80, 80, 80, 80, 80, 79, 80, 80, 80,
    79, 80, 80, 80, 79, 80, 80, 79, 79, 80, 80, 80, 79, 80, 80, 80, 79, 80, 80, 80, 79, 80, 80, 80, 79, 79, 80, 80, 79, 80, 80, 80,
    79, 80, 79, 79, 79, 80, 80, 79, 79, 79, 80, 80, 79, 80, 80, 80, 79, 80, 80, 80, 79, 80, 80, 79, 79, 79, 80, 79, 79, 79, 79, 80,
    79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 80, 79, 80, 79, 80, 79, 80, 79, 80, 79, 80, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79,
    79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 80, 79, 79, 79, 80, 79, 80, 79, 79, 79, 80, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79,
    79, 80, 80, 80, 79, 80, 80, 79, 79, 80, 80, 80, 79, 80, 80, 80, 79, 80, 80, 80, 79, 80, 80, 80, 79, 79, 80, 80, 79, 80, 80, 80,
    79, 80, 80, 80, 79, 80, 80, 79, 79, 79, 80, 80, 79, 80, 80, 80, 79, 80, 80, 80, 79, 80, 80, 79, 79, 79, 80, 80, 79, 79, 80, 80,
    79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79,
    79, 80, 80, 79, 79, 80, 80, 79, 79, 79, 80, 80, 79, 80, 80, 80, 79, 80, 80, 80, 79, 80, 80, 79, 79, 79, 80, 79, 79, 79, 79, 80,
    79, 80, 80, 80, 79, 80, 80, 79, 79, 79, 80, 80, 79, 80, 80, 80, 79, 80, 80, 80, 79, 80, 80, 79, 79, 79, 80, 80, 79, 79, 80, 80,
    79, 80, 80, 80, 79, 80, 80, 79, 79, 80, 80, 80, 79, 80, 80, 80, 79, 80, 80, 80, 79, 80, 80, 80, 79, 79, 80, 80, 79, 80, 80, 80
  },
  {
    73, 74, 74, 73, 73, 74, 74, 73, 73, 73, 74, 74, 73, 74, 74, 74, 73, 74, 74, 74, 73, 74, 74, 73, 73, 73, 74, 73, 73, 73, 73, 74,
    74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74,
    74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 73, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74,
    73, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 73, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74,
    73, 74, 74, 74, 73, 74, 74, 73, 73, 74, 74, 74, 73, 74, 74, 74, 73, 74, 74, 74, 73, 74, 74, 74, 73, 73, 74, 74, 73, 74, 74, 74,
    74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74,
    74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74,
    73, 74, 74, 74, 73, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 73, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 73, 74, 74, 74,
    73, 74, 74, 74, 73, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 73, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 73, 74, 74, 74,
    73, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 73, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74,
    74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74,
    74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74,
    73, 74, 74, 74, 73, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 73, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 73, 74, 74, 74,
    74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74,
    74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74,
    74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74,
    73, 74, 73, 73, 73, 74, 74, 73, 73, 73, 74, 74, 73, 74, 74, 74, 73, 74, 74, 74, 73, 74, 74, 73, 73, 73, 74, 73, 73, 73, 73, 74,
    74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74,
    74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74,
    74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74,
    73, 74, 74, 74, 73, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 73, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 73, 74, 74, 74,
    74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74,
    74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74,
    73, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 73, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74,
    73, 74, 74, 74, 73, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 73, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 73, 74, 74, 74,
    73, 74, 74, 74, 73, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 73, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 73, 74, 74, 74,
    74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74,
    73, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 73, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74,
    73, 74, 74, 74, 73, 74, 74, 73, 73, 74, 74, 74, 73, 74, 74, 74, 73, 74, 74, 74, 73, 74, 74, 74, 73, 73, 74, 74, 73, 74, 74, 74,
    73, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 73, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74,
    73, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 73, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74,
    74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74, 74
  },
  {
    64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
    64, 65, 65, 65, 64, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65,
    64, 65, 65, 65, 64, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 64, 65, 65, 65,
    64, 65, 65, 65, 64, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 64, 65, 65, 65,
    64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
    64, 65, 65, 65, 64, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 64, 65, 65, 65,
    64, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65,
    64, 64, 64, 64, 64, 64, 65, 64, 64, 64, 65, 65, 64, 65, 65, 65, 64, 65, 65, 65, 64, 65, 65, 64, 64, 64, 65, 64, 64, 64, 64, 64,
    64, 65, 65, 65, 64, 65, 65, 64, 64, 64, 65, 65, 64, 65, 65, 65, 64, 65, 65, 65, 64, 65, 65, 64, 64, 64, 65, 65, 64, 65, 65, 65,
    64, 65, 65, 65, 64, 65, 65, 64, 64, 65, 65, 65, 64, 65, 65, 65, 64, 65, 65, 65, 64, 65, 65, 65, 64, 64, 65, 65, 64, 65, 65, 65,
    64, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65,
    64, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65,
    64, 65, 65, 65, 64, 65, 65, 64, 64, 64, 65, 65, 64, 65, 65, 65, 64, 65, 65, 65, 64, 65, 65, 64, 64, 64, 65, 65, 64, 64, 65, 65,
    64, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65,
    64, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65,
    64, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65,
    64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
    64, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65,
    64, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65,
    64, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65,
    64, 65, 65, 65, 64, 65, 65, 64, 64, 64, 65, 65, 64, 65, 65, 65, 64, 65, 65, 65, 64, 65, 65, 64, 64, 64, 65, 65, 64, 65, 65, 65,
    64, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65,
    64, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65,
    64, 65, 65, 65, 64, 65, 65, 64, 64, 65, 65, 65, 64, 65, 65, 65, 64, 65, 65, 65, 64, 65, 65, 65, 64, 64, 65, 65, 64, 65, 65, 65,
    64, 65, 65, 65, 64, 65, 65, 64, 64, 64, 65, 65, 64, 65, 65, 65, 64, 65, 65, 65, 64, 65, 65, 64, 64, 64, 65, 65, 64, 65, 65, 65,
    64, 65, 64, 64, 64, 64, 65, 64, 64, 64, 65, 65, 64, 65, 65, 65, 64, 65, 65, 65, 64, 65, 65, 64, 64, 64, 65, 64, 64, 64, 64, 65,
    64, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65,
    64, 65, 65, 65, 64, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 64, 65, 65, 65,
    64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
    64, 65, 65, 65, 64, 65, 65, 64, 65, 65, 65, 65, 64, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 64, 65, 65, 65,
    64, 65, 65, 65, 64, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 64, 65, 65, 65,
    64, 65, 65, 65, 64, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65
  },
  {
    57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57,
    57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 58, 57, 58, 57, 58, 57, 58, 57, 58, 57, 58, 57, 57, 57, 57, 58, 57, 57, 57, 57, 57,
    57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 58, 57, 58, 57, 58, 57, 58, 57, 58, 57, 58, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57,
    57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 58, 57, 58, 57, 58, 57, 58, 57, 58, 57, 58, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57,
    57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57,
    57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 58, 57, 58, 57, 58, 57, 58, 57, 58, 57, 58, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57,
    57, 57, 57, 57, 57, 57, 58, 57, 57, 57, 58, 58, 57, 58, 58, 58, 57, 58, 58, 58, 57, 58, 58, 57, 57, 57, 58, 57, 57, 57, 57, 57,
    57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57,
    57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57,
    57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 58, 57, 58, 57, 58, 57, 58, 57, 58, 57, 58, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57,
    57, 57, 57, 57, 57, 57, 58, 57, 57, 57, 58, 58, 57, 58, 58, 58, 57, 58, 58, 58, 57, 58, 58, 57, 57, 57, 58, 57, 57, 57, 57, 57,
    57, 58, 58, 58, 57, 58, 58, 57, 57, 58, 58, 58, 57, 58, 58, 58, 57, 58, 58, 58, 57, 58, 58, 58, 57, 57, 58, 58, 57, 58, 58, 58,
    57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57,
    57, 58, 58, 58, 57, 58, 58, 57, 57, 58, 58, 58, 57, 58, 58, 58, 57, 58, 58, 58, 57, 58, 58, 58, 57, 57, 58, 58, 57, 58, 58, 58,
    57, 57, 57, 57, 57, 57, 58, 57, 57, 57, 58, 58, 57, 58, 58, 58, 57, 58, 58, 58, 57, 58, 58, 57, 57, 57, 58, 57, 57, 57, 57, 57,
    57, 58, 58, 58, 57, 58, 58, 57, 57, 58, 58, 58, 57, 58, 58, 58, 57, 58, 58, 58, 57, 58, 58, 58, 57, 57, 58, 58, 57, 58, 58, 58,
    57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57,
    57, 58, 58, 58, 57, 58, 58, 57, 57, 58, 58, 58, 57, 58, 58, 58, 57, 58, 58, 58, 57, 58, 58, 58, 57, 57, 58, 58, 57, 58, 58, 58,
    57, 57, 57, 57, 57, 57, 58, 57, 57, 57, 58, 58, 57, 58, 58, 58, 57, 58, 58, 58, 57, 58, 58, 57, 57, 57, 58, 57, 57, 57, 57, 57,
    57, 58, 58, 58, 57, 58, 58, 57, 57, 58, 58, 58, 57, 58, 58, 58, 57, 58, 58, 58, 57, 58, 58, 58, 57, 57, 58, 58, 57, 58, 58, 58,
    57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57,
    57, 58, 58, 58, 57, 58, 58, 57, 57, 58, 58, 58, 57, 58, 58, 58, 57, 58, 58, 58, 57, 58, 58, 58, 57, 57, 58, 58, 57, 58, 58, 58,
    57, 57, 57, 57, 57, 57, 58, 57, 57, 57, 58, 58, 57, 58, 58, 58, 57, 58, 58, 58, 57, 58, 58, 57, 57, 57, 58, 57, 57, 57, 57, 57,
    57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 58, 57, 58, 57, 58, 57, 58, 57, 58, 57, 58, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57,
    57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57,
    57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57,
    57, 58, 57, 57, 57, 57, 58, 57, 57, 57, 58, 58, 57, 58, 58, 58, 57, 58, 58, 58, 57, 58, 58, 57, 57, 57, 58, 57, 57, 57, 57, 57,
    57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 58, 57, 58, 57, 58, 57, 58, 57, 58, 57, 58, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57,
    57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57,
    57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 58, 57, 58, 57, 58, 57, 58, 57, 58, 57, 58, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57,
    57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 58, 57, 58, 57, 58, 57, 58, 57, 58, 57, 58, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57,
    57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 58, 57, 58, 57, 58, 57, 58, 57, 58, 57, 58, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57
  }
};
UInt g_aiDeQuantCoef1024[6][1024] =
{
  {
    40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40,
    40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40,
    40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40,
    40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40,
    40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40,
    40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40,
    40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40,
    40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40,
    40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40,
    40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40,
    40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40,
    40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 41, 40, 41, 40, 41, 40, 41, 40, 41, 40, 41, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40,
    40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40,
    40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 41, 40, 41, 40, 41, 40, 41, 40, 41, 40, 41, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40,
    40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40,
    40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 41, 40, 41, 40, 41, 40, 41, 40, 41, 40, 41, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40,
    40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40,
    40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 41, 40, 41, 40, 41, 40, 41, 40, 41, 40, 41, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40,
    40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40,
    40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 41, 40, 41, 40, 41, 40, 41, 40, 41, 40, 41, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40,
    40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40,
    40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 41, 40, 41, 40, 41, 40, 41, 40, 41, 40, 41, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40,
    40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40,
    40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40,
    40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40,
    40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40,
    40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40,
    40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40,
    40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40,
    40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40,
    40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40,
    40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40
  },
  {
    44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44,
    44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 45, 44, 45, 44, 45, 44, 45, 44, 45, 44, 45, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44,
    44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 45, 44, 45, 44, 44, 44, 45, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44,
    44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44,
    44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44,
    44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 45, 44, 45, 44, 45, 44, 45, 44, 45, 44, 45, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44,
    44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 45, 44, 45, 44, 45, 44, 45, 44, 45, 44, 45, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44,
    44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44,
    44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44,
    44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44,
    44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 45, 44, 45, 44, 45, 44, 45, 44, 45, 44, 45, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44,
    44, 45, 44, 44, 44, 45, 45, 44, 44, 44, 45, 45, 44, 45, 45, 45, 44, 45, 45, 45, 44, 45, 45, 44, 44, 44, 45, 44, 44, 44, 44, 45,
    44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44,
    44, 45, 44, 44, 44, 45, 45, 44, 44, 44, 45, 45, 44, 45, 45, 45, 44, 45, 45, 45, 44, 45, 45, 44, 44, 44, 45, 44, 44, 44, 44, 45,
    44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 45, 44, 45, 44, 45, 44, 45, 44, 45, 44, 45, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44,
    44, 45, 45, 44, 44, 45, 45, 44, 44, 44, 45, 45, 44, 45, 45, 45, 44, 45, 45, 45, 44, 45, 45, 44, 44, 44, 45, 44, 44, 44, 44, 45,
    44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44,
    44, 45, 45, 44, 44, 45, 45, 44, 44, 44, 45, 45, 44, 45, 45, 45, 44, 45, 45, 45, 44, 45, 45, 44, 44, 44, 45, 45, 44, 44, 45, 45,
    44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 45, 44, 45, 44, 45, 44, 45, 44, 45, 44, 45, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44,
    44, 45, 44, 44, 44, 45, 45, 44, 44, 44, 45, 45, 44, 45, 45, 45, 44, 45, 45, 45, 44, 45, 45, 44, 44, 44, 45, 44, 44, 44, 44, 45,
    44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44,
    44, 45, 45, 44, 44, 45, 45, 44, 44, 44, 45, 45, 44, 45, 45, 45, 44, 45, 45, 45, 44, 45, 45, 44, 44, 44, 45, 44, 44, 44, 44, 45,
    44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 45, 44, 45, 44, 45, 44, 45, 44, 45, 44, 45, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44,
    44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44,
    44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44,
    44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44,
    44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 45, 44, 45, 44, 45, 44, 45, 44, 45, 44, 45, 44, 44, 44, 44, 45, 44, 44, 44, 44, 44,
    44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 45, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44,
    44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44,
    44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44,
    44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 45, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44,
    44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 45, 44, 45, 44, 45, 44, 45, 44, 45, 44, 45, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44
  },
  {
    52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52,
    52, 53, 53, 53, 52, 53, 53, 52, 52, 52, 53, 53, 52, 53, 53, 53, 52, 53, 53, 53, 52, 53, 53, 52, 52, 52, 53, 53, 52, 52, 53, 53,
    52, 53, 52, 52, 52, 53, 53, 52, 52, 52, 53, 53, 52, 53, 53, 53, 52, 53, 53, 53, 52, 53, 53, 52, 52, 52, 53, 52, 52, 52, 52, 53,
    52, 53, 52, 52, 52, 52, 53, 52, 52, 52, 53, 53, 52, 53, 53, 53, 52, 53, 53, 53, 52, 53, 53, 52, 52, 52, 53, 52, 52, 52, 52, 52,
    52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52,
    52, 53, 53, 52, 52, 53, 53, 52, 52, 52, 53, 53, 52, 53, 53, 53, 52, 53, 53, 53, 52, 53, 53, 52, 52, 52, 53, 53, 52, 52, 53, 53,
    52, 53, 53, 53, 52, 53, 53, 52, 52, 53, 53, 53, 52, 53, 53, 53, 52, 53, 53, 53, 52, 53, 53, 53, 52, 52, 53, 53, 52, 53, 53, 53,
    52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52,
    52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 53, 52, 53, 52, 53, 52, 53, 52, 53, 52, 53, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52,
    52, 52, 52, 52, 52, 52, 53, 52, 52, 52, 53, 53, 52, 53, 53, 53, 52, 53, 53, 53, 52, 53, 53, 52, 52, 52, 53, 52, 52, 52, 52, 52,
    52, 53, 53, 53, 52, 53, 53, 52, 52, 53, 53, 53, 52, 53, 53, 53, 52, 53, 53, 53, 52, 53, 53, 53, 52, 52, 53, 53, 52, 53, 53, 53,
    52, 53, 53, 53, 52, 53, 53, 52, 53, 53, 53, 53, 53, 53, 53, 53, 52, 53, 53, 53, 53, 53, 53, 53, 53, 52, 53, 53, 52, 53, 53, 53,
    52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 53, 52, 53, 52, 53, 52, 53, 52, 53, 52, 53, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52,
    52, 53, 53, 53, 52, 53, 53, 52, 53, 53, 53, 53, 53, 53, 53, 53, 52, 53, 53, 53, 53, 53, 53, 53, 53, 52, 53, 53, 52, 53, 53, 53,
    52, 53, 53, 53, 52, 53, 53, 52, 52, 53, 53, 53, 52, 53, 53, 53, 52, 53, 53, 53, 52, 53, 53, 53, 52, 52, 53, 53, 52, 53, 53, 53,
    52, 53, 53, 53, 52, 53, 53, 52, 53, 53, 53, 53, 53, 53, 53, 53, 52, 53, 53, 53, 53, 53, 53, 53, 53, 52, 53, 53, 52, 53, 53, 53,
    52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52,
    52, 53, 53, 53, 52, 53, 53, 52, 53, 53, 53, 53, 53, 53, 53, 53, 52, 53, 53, 53, 53, 53, 53, 53, 53, 52, 53, 53, 52, 53, 53, 53,
    52, 53, 53, 53, 52, 53, 53, 52, 52, 53, 53, 53, 52, 53, 53, 53, 52, 53, 53, 53, 52, 53, 53, 53, 52, 52, 53, 53, 52, 53, 53, 53,
    52, 53, 53, 53, 52, 53, 53, 52, 53, 53, 53, 53, 53, 53, 53, 53, 52, 53, 53, 53, 53, 53, 53, 53, 53, 52, 53, 53, 52, 53, 53, 53,
    52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 53, 52, 53, 52, 53, 52, 53, 52, 53, 52, 53, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52,
    52, 53, 53, 53, 52, 53, 53, 52, 53, 53, 53, 53, 53, 53, 53, 53, 52, 53, 53, 53, 53, 53, 53, 53, 53, 52, 53, 53, 52, 53, 53, 53,
    52, 53, 53, 53, 52, 53, 53, 52, 52, 53, 53, 53, 52, 53, 53, 53, 52, 53, 53, 53, 52, 53, 53, 53, 52, 52, 53, 53, 52, 53, 53, 53,
    52, 52, 52, 52, 52, 52, 53, 52, 52, 52, 53, 53, 52, 53, 53, 53, 52, 53, 53, 53, 52, 53, 53, 52, 52, 52, 53, 52, 52, 52, 52, 52,
    52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 53, 52, 53, 52, 53, 52, 53, 52, 53, 52, 53, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52,
    52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52,
    52, 53, 53, 53, 52, 53, 53, 52, 52, 53, 53, 53, 52, 53, 53, 53, 52, 53, 53, 53, 52, 53, 53, 53, 52, 52, 53, 53, 52, 53, 53, 53,
    52, 53, 52, 52, 52, 53, 53, 52, 52, 52, 53, 53, 52, 53, 53, 53, 52, 53, 53, 53, 52, 53, 53, 52, 52, 52, 53, 52, 52, 52, 52, 53,
    52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52, 52,
    52, 52, 52, 52, 52, 52, 53, 52, 52, 52, 53, 53, 52, 53, 53, 53, 52, 53, 53, 53, 52, 53, 53, 52, 52, 52, 53, 52, 52, 52, 52, 52,
    52, 53, 52, 52, 52, 53, 53, 52, 52, 52, 53, 53, 52, 53, 53, 53, 52, 53, 53, 53, 52, 53, 53, 52, 52, 52, 53, 52, 52, 52, 52, 53,
    52, 53, 53, 52, 52, 53, 53, 52, 52, 52, 53, 53, 52, 53, 53, 53, 52, 53, 53, 53, 52, 53, 53, 52, 52, 52, 53, 53, 52, 52, 53, 53
  },
  {
    56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56,
    56, 57, 57, 57, 56, 57, 57, 56, 56, 57, 57, 57, 56, 57, 57, 57, 56, 57, 57, 57, 56, 57, 57, 57, 56, 56, 57, 57, 56, 57, 57, 57,
    56, 57, 57, 57, 56, 57, 57, 56, 56, 57, 57, 57, 56, 57, 57, 57, 56, 57, 57, 57, 56, 57, 57, 57, 56, 56, 57, 57, 56, 57, 57, 57,
    56, 57, 57, 57, 56, 57, 57, 56, 56, 57, 57, 57, 56, 57, 57, 57, 56, 57, 57, 57, 56, 57, 57, 57, 56, 56, 57, 57, 56, 57, 57, 57,
    56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56,
    56, 57, 57, 57, 56, 57, 57, 56, 56, 57, 57, 57, 56, 57, 57, 57, 56, 57, 57, 57, 56, 57, 57, 57, 56, 56, 57, 57, 56, 57, 57, 57,
    56, 57, 57, 57, 56, 57, 57, 56, 57, 57, 57, 57, 56, 57, 57, 57, 56, 57, 57, 57, 57, 57, 57, 57, 57, 56, 57, 57, 56, 57, 57, 57,
    56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 57, 56, 57, 56, 57, 56, 57, 56, 57, 56, 57, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56,
    56, 56, 56, 56, 56, 56, 57, 56, 56, 56, 56, 57, 56, 57, 56, 57, 56, 57, 56, 57, 56, 57, 56, 56, 56, 56, 57, 56, 56, 56, 56, 56,
    56, 57, 57, 57, 56, 57, 57, 56, 56, 56, 57, 57, 56, 57, 57, 57, 56, 57, 57, 57, 56, 57, 57, 56, 56, 56, 57, 57, 56, 56, 57, 57,
    56, 57, 57, 57, 56, 57, 57, 56, 56, 57, 57, 57, 56, 57, 57, 57, 56, 57, 57, 57, 57, 57, 57, 57, 57, 56, 57, 57, 56, 57, 57, 57,
    56, 57, 57, 57, 56, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 56, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 56, 57, 57, 57,
    56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 57, 56, 57, 56, 57, 56, 57, 56, 57, 56, 57, 56, 56, 56, 56, 57, 56, 56, 56, 56, 56,
    56, 57, 57, 57, 56, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 56, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 56, 57, 57, 57,
    56, 57, 57, 57, 56, 57, 57, 56, 56, 57, 57, 57, 56, 57, 57, 57, 56, 57, 57, 57, 56, 57, 57, 57, 56, 56, 57, 57, 56, 57, 57, 57,
    56, 57, 57, 57, 56, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 56, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 56, 57, 57, 57,
    56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56,
    56, 57, 57, 57, 56, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 56, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 56, 57, 57, 57,
    56, 57, 57, 57, 56, 57, 57, 56, 56, 57, 57, 57, 56, 57, 57, 57, 56, 57, 57, 57, 56, 57, 57, 57, 56, 56, 57, 57, 56, 57, 57, 57,
    56, 57, 57, 57, 56, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 56, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 56, 57, 57, 57,
    56, 56, 56, 56, 56, 56, 57, 56, 56, 56, 57, 57, 56, 57, 56, 57, 56, 57, 56, 57, 56, 57, 56, 56, 56, 56, 57, 56, 56, 56, 56, 56,
    56, 57, 57, 57, 56, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 56, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 56, 57, 57, 57,
    56, 57, 57, 57, 56, 57, 57, 56, 56, 57, 57, 57, 56, 57, 57, 57, 56, 57, 57, 57, 56, 57, 57, 57, 56, 56, 57, 57, 56, 57, 57, 57,
    56, 57, 57, 57, 56, 57, 57, 56, 56, 56, 57, 57, 56, 57, 57, 57, 56, 57, 57, 57, 56, 57, 57, 56, 56, 56, 57, 57, 56, 57, 57, 57,
    56, 56, 56, 56, 56, 56, 57, 56, 56, 56, 57, 57, 56, 57, 56, 57, 56, 57, 56, 57, 56, 57, 56, 56, 56, 56, 57, 56, 56, 56, 56, 56,
    56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 57, 56, 57, 56, 57, 56, 57, 56, 57, 56, 57, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56,
    56, 57, 57, 57, 56, 57, 57, 56, 57, 57, 57, 57, 57, 57, 57, 57, 56, 57, 57, 57, 57, 57, 57, 57, 57, 56, 57, 57, 56, 57, 57, 57,
    56, 57, 57, 57, 56, 57, 57, 56, 56, 57, 57, 57, 56, 57, 57, 57, 56, 57, 57, 57, 56, 57, 57, 57, 56, 56, 57, 57, 56, 57, 57, 57,
    56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56,
    56, 57, 57, 57, 56, 57, 57, 56, 56, 56, 57, 57, 56, 57, 57, 57, 56, 57, 57, 57, 56, 57, 57, 57, 56, 56, 57, 57, 56, 57, 57, 57,
    56, 57, 57, 57, 56, 57, 57, 56, 56, 57, 57, 57, 56, 57, 57, 57, 56, 57, 57, 57, 56, 57, 57, 57, 56, 56, 57, 57, 56, 57, 57, 57,
    56, 57, 57, 57, 56, 57, 57, 56, 56, 57, 57, 57, 56, 57, 57, 57, 56, 57, 57, 57, 56, 57, 57, 57, 56, 56, 57, 57, 56, 57, 57, 57
  },
  {
    64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
    64, 65, 65, 65, 64, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65,
    64, 65, 65, 65, 64, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 64, 65, 65, 65,
    64, 65, 65, 65, 64, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 64, 65, 65, 65,
    64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
    64, 65, 65, 65, 64, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 64, 65, 65, 65,
    64, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65,
    64, 64, 64, 64, 64, 64, 65, 64, 64, 64, 65, 65, 64, 65, 65, 65, 64, 65, 65, 65, 64, 65, 65, 64, 64, 64, 65, 64, 64, 64, 64, 64,
    64, 65, 65, 65, 64, 65, 65, 64, 64, 64, 65, 65, 64, 65, 65, 65, 64, 65, 65, 65, 64, 65, 65, 64, 64, 64, 65, 65, 64, 65, 65, 65,
    64, 65, 65, 65, 64, 65, 65, 64, 64, 65, 65, 65, 64, 65, 65, 65, 64, 65, 65, 65, 64, 65, 65, 65, 64, 64, 65, 65, 64, 65, 65, 65,
    64, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65,
    64, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65,
    64, 65, 65, 65, 64, 65, 65, 64, 64, 64, 65, 65, 64, 65, 65, 65, 64, 65, 65, 65, 64, 65, 65, 64, 64, 64, 65, 65, 64, 64, 65, 65,
    64, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65,
    64, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65,
    64, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65,
    64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
    64, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65,
    64, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65,
    64, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65,
    64, 65, 65, 65, 64, 65, 65, 64, 64, 64, 65, 65, 64, 65, 65, 65, 64, 65, 65, 65, 64, 65, 65, 64, 64, 64, 65, 65, 64, 65, 65, 65,
    64, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65,
    64, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65,
    64, 65, 65, 65, 64, 65, 65, 64, 64, 65, 65, 65, 64, 65, 65, 65, 64, 65, 65, 65, 64, 65, 65, 65, 64, 64, 65, 65, 64, 65, 65, 65,
    64, 65, 65, 65, 64, 65, 65, 64, 64, 64, 65, 65, 64, 65, 65, 65, 64, 65, 65, 65, 64, 65, 65, 64, 64, 64, 65, 65, 64, 65, 65, 65,
    64, 65, 64, 64, 64, 64, 65, 64, 64, 64, 65, 65, 64, 65, 65, 65, 64, 65, 65, 65, 64, 65, 65, 64, 64, 64, 65, 64, 64, 64, 64, 65,
    64, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65,
    64, 65, 65, 65, 64, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 64, 65, 65, 65,
    64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
    64, 65, 65, 65, 64, 65, 65, 64, 65, 65, 65, 65, 64, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 64, 65, 65, 65,
    64, 65, 65, 65, 64, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 64, 65, 65, 65,
    64, 65, 65, 65, 64, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 64, 65, 65, 65
  },
  {
    72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72,
    72, 73, 73, 73, 72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 72, 73, 73, 73,
    72, 73, 73, 73, 72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 72, 73, 73, 73,
    72, 73, 73, 73, 72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 72, 73, 73, 73,
    72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 73, 72, 73, 72, 73, 72, 73, 72, 73, 72, 73, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72,
    72, 73, 73, 73, 72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 72, 73, 73, 73,
    72, 73, 73, 73, 72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 72, 73, 73, 73,
    72, 73, 73, 73, 72, 73, 73, 72, 72, 73, 73, 73, 72, 73, 73, 73, 72, 73, 73, 73, 72, 73, 73, 73, 72, 72, 73, 73, 72, 73, 73, 73,
    72, 73, 73, 73, 72, 73, 73, 72, 72, 73, 73, 73, 72, 73, 73, 73, 72, 73, 73, 73, 72, 73, 73, 73, 72, 72, 73, 73, 72, 73, 73, 73,
    72, 73, 73, 73, 72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 72, 73, 73, 73,
    72, 73, 73, 73, 72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 72, 73, 73, 73,
    72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73,
    72, 73, 73, 73, 72, 73, 73, 72, 72, 73, 73, 73, 72, 73, 73, 73, 72, 73, 73, 73, 72, 73, 73, 73, 72, 72, 73, 73, 72, 73, 73, 73,
    72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73,
    72, 73, 73, 73, 72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 72, 73, 73, 73,
    72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73,
    72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72,
    72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73,
    72, 73, 73, 73, 72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 72, 73, 73, 73,
    72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73,
    72, 73, 73, 73, 72, 73, 73, 72, 72, 73, 73, 73, 72, 73, 73, 73, 72, 73, 73, 73, 72, 73, 73, 73, 72, 72, 73, 73, 72, 73, 73, 73,
    72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73,
    72, 73, 73, 73, 72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 72, 73, 73, 73,
    72, 73, 73, 73, 72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 72, 73, 73, 73,
    72, 73, 73, 73, 72, 73, 73, 72, 72, 73, 73, 73, 72, 73, 73, 73, 72, 73, 73, 73, 72, 73, 73, 73, 72, 72, 73, 73, 72, 73, 73, 73,
    72, 73, 73, 73, 72, 73, 73, 72, 72, 73, 73, 73, 72, 73, 73, 73, 72, 73, 73, 73, 72, 73, 73, 73, 72, 72, 73, 73, 72, 73, 73, 73,
    72, 73, 73, 73, 72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 72, 73, 73, 73,
    72, 73, 73, 73, 72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 72, 73, 73, 73,
    72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 73, 72, 73, 72, 73, 72, 73, 72, 73, 72, 73, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72,
    72, 73, 73, 73, 72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 72, 73, 73, 73,
    72, 73, 73, 73, 72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 72, 73, 73, 73,
    72, 73, 73, 73, 72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 72, 73, 73, 73
  }
};

UInt g_aiQuantCoef256[6][256] =
{
  {
    207, 209, 209, 208, 207, 209, 208, 209, 207, 209, 208, 209, 207, 208, 209, 209,
    209, 212, 212, 211, 209, 211, 210, 212, 209, 212, 210, 211, 209, 211, 212, 212,
    209, 212, 212, 211, 209, 211, 210, 212, 209, 212, 210, 211, 209, 211, 212, 212,
    208, 211, 211, 210, 208, 210, 209, 211, 208, 211, 209, 210, 208, 210, 211, 211,
    207, 209, 209, 208, 207, 208, 208, 209, 207, 209, 208, 208, 207, 208, 209, 209,
    209, 211, 211, 210, 208, 210, 210, 211, 209, 211, 210, 210, 208, 210, 211, 211,
    208, 210, 210, 209, 208, 210, 209, 210, 208, 211, 209, 210, 208, 209, 210, 210,
    209, 212, 212, 211, 209, 211, 210, 212, 209, 212, 210, 211, 209, 211, 212, 212,
    207, 209, 209, 208, 207, 209, 208, 209, 207, 210, 208, 209, 207, 208, 209, 209,
    209, 212, 212, 211, 209, 211, 211, 212, 210, 212, 210, 211, 209, 211, 212, 212,
    208, 210, 210, 209, 208, 210, 209, 210, 208, 210, 209, 210, 208, 209, 210, 210,
    209, 211, 211, 210, 208, 210, 210, 211, 209, 211, 210, 210, 208, 210, 211, 211,
    207, 209, 209, 208, 207, 208, 208, 209, 207, 209, 208, 208, 207, 208, 209, 209,
    208, 211, 211, 210, 208, 210, 209, 211, 208, 211, 209, 210, 208, 210, 211, 211,
    209, 212, 212, 211, 209, 211, 210, 212, 209, 212, 210, 211, 209, 211, 212, 212,
    209, 212, 212, 211, 209, 211, 210, 212, 209, 212, 210, 211, 209, 211, 212, 212
  },
  {
    188, 190, 190, 189, 188, 190, 189, 190, 188, 190, 189, 190, 188, 189, 190, 190,
    190, 192, 192, 191, 190, 192, 191, 192, 190, 193, 191, 192, 190, 191, 192, 192,
    190, 192, 192, 191, 190, 192, 191, 192, 190, 192, 191, 192, 190, 191, 192, 192,
    189, 191, 191, 190, 189, 191, 190, 191, 189, 192, 190, 191, 189, 190, 191, 191,
    188, 190, 190, 189, 188, 189, 189, 190, 188, 190, 189, 189, 188, 189, 190, 190,
    190, 192, 192, 191, 189, 191, 191, 192, 190, 192, 191, 191, 189, 191, 192, 192,
    189, 191, 191, 190, 189, 191, 190, 191, 189, 191, 190, 191, 189, 190, 191, 191,
    190, 192, 192, 191, 190, 192, 191, 193, 190, 193, 191, 192, 190, 191, 192, 192,
    188, 190, 190, 189, 188, 190, 189, 190, 188, 190, 189, 190, 188, 189, 190, 190,
    190, 193, 192, 192, 190, 192, 191, 193, 190, 193, 191, 192, 190, 192, 192, 193,
    189, 191, 191, 190, 189, 191, 190, 191, 189, 191, 190, 191, 189, 190, 191, 191,
    190, 192, 192, 191, 189, 191, 191, 192, 190, 192, 191, 191, 189, 191, 192, 192,
    188, 190, 190, 189, 188, 189, 189, 190, 188, 190, 189, 189, 188, 189, 190, 190,
    189, 191, 191, 190, 189, 191, 190, 191, 189, 192, 190, 191, 189, 190, 191, 191,
    190, 192, 192, 191, 190, 192, 191, 192, 190, 192, 191, 192, 190, 191, 192, 192,
    190, 192, 192, 191, 190, 192, 191, 192, 190, 193, 191, 192, 190, 191, 192, 192
  },
  {
    159, 161, 161, 160, 159, 160, 160, 161, 159, 161, 160, 160, 159, 160, 161, 161,
    161, 163, 163, 162, 161, 162, 162, 163, 161, 163, 162, 162, 161, 162, 163, 163,
    161, 163, 163, 162, 161, 162, 162, 163, 161, 163, 162, 162, 161, 162, 163, 163,
    160, 162, 162, 161, 160, 161, 161, 162, 160, 162, 161, 161, 160, 161, 162, 162,
    159, 161, 161, 160, 159, 160, 160, 161, 159, 161, 160, 160, 159, 160, 161, 161,
    160, 162, 162, 161, 160, 162, 161, 162, 160, 162, 161, 162, 160, 161, 162, 162,
    160, 162, 162, 161, 160, 161, 161, 162, 160, 162, 161, 161, 160, 161, 162, 162,
    161, 163, 163, 162, 161, 162, 162, 163, 161, 163, 162, 162, 161, 162, 163, 163,
    159, 161, 161, 160, 159, 160, 160, 161, 159, 161, 160, 160, 159, 160, 161, 161,
    161, 163, 163, 162, 161, 162, 162, 163, 161, 163, 162, 162, 161, 162, 163, 163,
    160, 162, 162, 161, 160, 161, 161, 162, 160, 162, 161, 161, 160, 161, 162, 162,
    160, 162, 162, 161, 160, 162, 161, 162, 160, 162, 161, 162, 160, 161, 162, 162,
    159, 161, 161, 160, 159, 160, 160, 161, 159, 161, 160, 160, 159, 160, 161, 161,
    160, 162, 162, 161, 160, 161, 161, 162, 160, 162, 161, 161, 160, 161, 162, 162,
    161, 163, 163, 162, 161, 162, 162, 163, 161, 163, 162, 162, 161, 162, 163, 163,
    161, 163, 163, 162, 161, 162, 162, 163, 161, 163, 162, 162, 161, 162, 163, 163
  },
  {
    148, 150, 150, 149, 148, 149, 149, 150, 148, 150, 149, 149, 148, 149, 150, 150,
    150, 151, 151, 150, 149, 151, 150, 151, 150, 151, 150, 151, 149, 150, 151, 151,
    150, 151, 151, 150, 149, 151, 150, 151, 150, 151, 150, 151, 149, 150, 151, 151,
    149, 150, 150, 150, 149, 150, 150, 150, 149, 150, 150, 150, 149, 150, 150, 150,
    148, 149, 149, 149, 148, 149, 149, 149, 148, 149, 149, 149, 148, 149, 149, 149,
    149, 151, 151, 150, 149, 150, 150, 151, 149, 151, 150, 150, 149, 150, 151, 151,
    149, 150, 150, 150, 149, 150, 149, 150, 149, 150, 149, 150, 149, 150, 150, 150,
    150, 151, 151, 150, 149, 151, 150, 151, 150, 151, 150, 151, 149, 150, 151, 151,
    148, 150, 150, 149, 148, 149, 149, 150, 148, 150, 149, 149, 148, 149, 150, 150,
    150, 151, 151, 150, 149, 151, 150, 151, 150, 151, 150, 151, 149, 150, 151, 151,
    149, 150, 150, 150, 149, 150, 149, 150, 149, 150, 149, 150, 149, 150, 150, 150,
    149, 151, 151, 150, 149, 150, 150, 151, 149, 151, 150, 150, 149, 150, 151, 151,
    148, 149, 149, 149, 148, 149, 149, 149, 148, 149, 149, 149, 148, 149, 149, 149,
    149, 150, 150, 150, 149, 150, 150, 150, 149, 150, 150, 150, 149, 150, 150, 150,
    150, 151, 151, 150, 149, 151, 150, 151, 150, 151, 150, 151, 149, 150, 151, 151,
    150, 151, 151, 150, 149, 151, 150, 151, 150, 151, 150, 151, 149, 150, 151, 151
  },
  {
    129, 131, 131, 130, 129, 130, 130, 131, 129, 131, 130, 130, 129, 130, 131, 131,
    131, 132, 132, 132, 131, 132, 131, 132, 131, 132, 131, 132, 131, 132, 132, 132,
    131, 132, 132, 132, 131, 132, 131, 132, 131, 132, 131, 132, 131, 132, 132, 132,
    130, 132, 132, 131, 130, 131, 131, 132, 130, 132, 131, 131, 130, 131, 132, 132,
    129, 131, 131, 130, 129, 130, 130, 131, 129, 131, 130, 130, 129, 130, 131, 131,
    130, 132, 132, 131, 130, 131, 131, 132, 130, 132, 131, 131, 130, 131, 132, 132,
    130, 131, 131, 131, 130, 131, 131, 132, 130, 132, 131, 131, 130, 131, 131, 131,
    131, 132, 132, 132, 131, 132, 132, 132, 131, 132, 132, 132, 131, 132, 132, 132,
    129, 131, 131, 130, 129, 130, 130, 131, 129, 131, 130, 130, 129, 130, 131, 131,
    131, 132, 132, 132, 131, 132, 132, 132, 131, 132, 132, 132, 131, 132, 132, 132,
    130, 131, 131, 131, 130, 131, 131, 132, 130, 132, 131, 131, 130, 131, 131, 131,
    130, 132, 132, 131, 130, 131, 131, 132, 130, 132, 131, 131, 130, 131, 132, 132,
    129, 131, 131, 130, 129, 130, 130, 131, 129, 131, 130, 130, 129, 130, 131, 131,
    130, 132, 132, 131, 130, 131, 131, 132, 130, 132, 131, 131, 130, 131, 132, 132,
    131, 132, 132, 132, 131, 132, 131, 132, 131, 132, 131, 132, 131, 132, 132, 132,
    131, 132, 132, 132, 131, 132, 131, 132, 131, 132, 131, 132, 131, 132, 132, 132
  },
  {
    115, 116, 116, 116, 115, 116, 116, 116, 115, 116, 116, 116, 115, 116, 116, 116,
    116, 118, 118, 117, 116, 117, 117, 118, 116, 118, 117, 117, 116, 117, 118, 118,
    116, 118, 118, 117, 116, 117, 117, 118, 116, 118, 117, 117, 116, 117, 118, 118,
    116, 117, 117, 116, 116, 117, 116, 117, 116, 117, 116, 117, 116, 116, 117, 117,
    115, 116, 116, 116, 115, 116, 116, 116, 115, 116, 116, 116, 115, 116, 116, 116,
    116, 117, 117, 117, 116, 117, 116, 117, 116, 117, 116, 117, 116, 117, 117, 117,
    116, 117, 117, 116, 116, 116, 116, 117, 116, 117, 116, 116, 116, 116, 117, 117,
    116, 118, 118, 117, 116, 117, 117, 118, 116, 118, 117, 117, 116, 117, 118, 118,
    115, 116, 116, 116, 115, 116, 116, 116, 115, 116, 116, 116, 115, 116, 116, 116,
    116, 118, 118, 117, 116, 117, 117, 118, 116, 118, 117, 117, 116, 117, 118, 118,
    116, 117, 117, 116, 116, 116, 116, 117, 116, 117, 116, 116, 116, 116, 117, 117,
    116, 117, 117, 117, 116, 117, 116, 117, 116, 117, 116, 117, 116, 117, 117, 117,
    115, 116, 116, 116, 115, 116, 116, 116, 115, 116, 116, 116, 115, 116, 116, 116,
    116, 117, 117, 116, 116, 117, 116, 117, 116, 117, 116, 117, 116, 116, 117, 117,
    116, 118, 118, 117, 116, 117, 117, 118, 116, 118, 117, 117, 116, 117, 118, 118,
    116, 118, 118, 117, 116, 117, 117, 118, 116, 118, 117, 117, 116, 117, 118, 118
  }
  
};
UInt g_aiDeQuantCoef256[6][256] =
{
  {
    81, 82, 82, 81, 81, 81, 81, 82, 81, 82, 81, 81, 81, 81, 82, 82,
    82, 83, 83, 82, 82, 82, 82, 83, 82, 83, 82, 82, 82, 82, 83, 83,
    82, 83, 83, 82, 82, 82, 82, 83, 82, 83, 82, 82, 82, 82, 83, 83,
    81, 82, 82, 82, 81, 82, 82, 82, 81, 82, 82, 82, 81, 82, 82, 82,
    81, 82, 82, 81, 81, 81, 81, 82, 81, 82, 81, 81, 81, 81, 82, 82,
    81, 82, 82, 82, 81, 82, 82, 82, 81, 82, 82, 82, 81, 82, 82, 82,
    81, 82, 82, 82, 81, 82, 82, 82, 81, 82, 82, 82, 81, 82, 82, 82,
    82, 83, 83, 82, 82, 82, 82, 83, 82, 83, 82, 82, 82, 82, 83, 83,
    81, 82, 82, 81, 81, 81, 81, 82, 81, 82, 81, 81, 81, 81, 82, 82,
    82, 83, 83, 82, 82, 82, 82, 83, 82, 83, 82, 82, 82, 82, 83, 83,
    81, 82, 82, 82, 81, 82, 82, 82, 81, 82, 82, 82, 81, 82, 82, 82,
    81, 82, 82, 82, 81, 82, 82, 82, 81, 82, 82, 82, 81, 82, 82, 82,
    81, 82, 82, 81, 81, 81, 81, 82, 81, 82, 81, 81, 81, 81, 82, 82,
    81, 82, 82, 82, 81, 82, 82, 82, 81, 82, 82, 82, 81, 82, 82, 82,
    82, 83, 83, 82, 82, 82, 82, 83, 82, 83, 82, 82, 82, 82, 83, 83,
    82, 83, 83, 82, 82, 82, 82, 83, 82, 83, 82, 82, 82, 82, 83, 83
  },
  {
    89, 90, 90, 90, 89, 90, 89, 90, 89, 90, 89, 90, 89, 90, 90, 90,
    90, 91, 91, 90, 90, 91, 90, 91, 90, 91, 90, 91, 90, 90, 91, 91,
    90, 91, 91, 90, 90, 91, 90, 91, 90, 91, 90, 91, 90, 90, 91, 91,
    90, 90, 90, 90, 89, 90, 90, 91, 90, 91, 90, 90, 89, 90, 90, 90,
    89, 90, 90, 89, 89, 90, 89, 90, 89, 90, 89, 90, 89, 89, 90, 90,
    90, 91, 91, 90, 90, 90, 90, 91, 90, 91, 90, 90, 90, 90, 91, 91,
    89, 90, 90, 90, 89, 90, 90, 90, 89, 90, 90, 90, 89, 90, 90, 90,
    90, 91, 91, 91, 90, 91, 90, 91, 90, 91, 90, 91, 90, 91, 91, 91,
    89, 90, 90, 90, 89, 90, 89, 90, 89, 90, 89, 90, 89, 90, 90, 90,
    90, 91, 91, 91, 90, 91, 90, 91, 90, 91, 90, 91, 90, 91, 91, 91,
    89, 90, 90, 90, 89, 90, 90, 90, 89, 90, 90, 90, 89, 90, 90, 90,
    90, 91, 91, 90, 90, 90, 90, 91, 90, 91, 90, 90, 90, 90, 91, 91,
    89, 90, 90, 89, 89, 90, 89, 90, 89, 90, 89, 90, 89, 89, 90, 90,
    90, 90, 90, 90, 89, 90, 90, 91, 90, 91, 90, 90, 89, 90, 90, 90,
    90, 91, 91, 90, 90, 91, 90, 91, 90, 91, 90, 91, 90, 90, 91, 91,
    90, 91, 91, 90, 90, 91, 90, 91, 90, 91, 90, 91, 90, 90, 91, 91
  },
  {
    105, 106, 106, 106, 105, 106, 106, 106, 105, 106, 106, 106, 105, 106, 106, 106,
    106, 107, 107, 107, 106, 107, 107, 108, 106, 108, 107, 107, 106, 107, 107, 107,
    106, 107, 107, 107, 106, 107, 107, 107, 106, 108, 107, 107, 106, 107, 107, 107,
    106, 107, 107, 106, 106, 107, 106, 107, 106, 107, 106, 107, 106, 106, 107, 107,
    105, 106, 106, 106, 105, 106, 106, 106, 105, 106, 106, 106, 105, 106, 106, 106,
    106, 107, 107, 107, 106, 107, 106, 107, 106, 107, 106, 107, 106, 107, 107, 107,
    106, 107, 107, 106, 106, 106, 106, 107, 106, 107, 106, 106, 106, 106, 107, 107,
    106, 108, 107, 107, 106, 107, 107, 108, 106, 108, 107, 107, 106, 107, 107, 108,
    105, 106, 106, 106, 105, 106, 106, 106, 105, 106, 106, 106, 105, 106, 106, 106,
    106, 108, 108, 107, 106, 107, 107, 108, 106, 108, 107, 107, 106, 107, 108, 108,
    106, 107, 107, 106, 106, 106, 106, 107, 106, 107, 106, 106, 106, 106, 107, 107,
    106, 107, 107, 107, 106, 107, 106, 107, 106, 107, 106, 107, 106, 107, 107, 107,
    105, 106, 106, 106, 105, 106, 106, 106, 105, 106, 106, 106, 105, 106, 106, 106,
    106, 107, 107, 106, 106, 107, 106, 107, 106, 107, 106, 107, 106, 106, 107, 107,
    106, 107, 107, 107, 106, 107, 107, 107, 106, 108, 107, 107, 106, 107, 107, 107,
    106, 107, 107, 107, 106, 107, 107, 108, 106, 108, 107, 107, 106, 107, 107, 107
  },
  {
    113, 115, 114, 114, 113, 114, 114, 115, 113, 115, 114, 114, 113, 114, 114, 114,
    115, 116, 116, 115, 114, 115, 115, 116, 115, 116, 115, 115, 114, 115, 116, 116,
    114, 116, 116, 115, 114, 115, 115, 116, 114, 116, 115, 115, 114, 115, 116, 116,
    114, 115, 115, 115, 114, 115, 114, 115, 114, 115, 114, 115, 114, 115, 115, 115,
    113, 114, 114, 114, 113, 114, 114, 114, 113, 114, 114, 114, 113, 114, 114, 114,
    114, 115, 115, 115, 114, 115, 115, 115, 114, 115, 115, 115, 114, 115, 115, 115,
    114, 115, 115, 114, 114, 115, 114, 115, 114, 115, 114, 115, 114, 114, 115, 115,
    115, 116, 116, 115, 114, 115, 115, 116, 115, 116, 115, 115, 114, 115, 116, 116,
    113, 115, 114, 114, 113, 114, 114, 115, 113, 115, 114, 114, 113, 114, 114, 115,
    115, 116, 116, 115, 114, 115, 115, 116, 115, 116, 115, 115, 114, 115, 116, 116,
    114, 115, 115, 114, 114, 115, 114, 115, 114, 115, 114, 115, 114, 114, 115, 115,
    114, 115, 115, 115, 114, 115, 115, 115, 114, 115, 115, 115, 114, 115, 115, 115,
    113, 114, 114, 114, 113, 114, 114, 114, 113, 114, 114, 114, 113, 114, 114, 114,
    114, 115, 115, 115, 114, 115, 114, 115, 114, 115, 114, 115, 114, 115, 115, 115,
    114, 116, 116, 115, 114, 115, 115, 116, 114, 116, 115, 115, 114, 115, 116, 116,
    114, 116, 116, 115, 114, 115, 115, 116, 115, 116, 115, 115, 114, 115, 116, 116
  },
  {
    129, 131, 131, 130, 129, 130, 130, 131, 129, 131, 130, 130, 129, 130, 131, 131,
    131, 132, 132, 132, 131, 132, 131, 132, 131, 132, 131, 132, 131, 132, 132, 132,
    131, 132, 132, 132, 131, 132, 131, 132, 131, 132, 131, 132, 131, 132, 132, 132,
    130, 132, 132, 131, 130, 131, 131, 132, 130, 132, 131, 131, 130, 131, 132, 132,
    129, 131, 131, 130, 129, 130, 130, 131, 129, 131, 130, 130, 129, 130, 131, 131,
    130, 132, 132, 131, 130, 131, 131, 132, 130, 132, 131, 131, 130, 131, 132, 132,
    130, 131, 131, 131, 130, 131, 131, 132, 130, 132, 131, 131, 130, 131, 131, 131,
    131, 132, 132, 132, 131, 132, 132, 132, 131, 132, 132, 132, 131, 132, 132, 132,
    129, 131, 131, 130, 129, 130, 130, 131, 129, 131, 130, 130, 129, 130, 131, 131,
    131, 132, 132, 132, 131, 132, 132, 132, 131, 132, 132, 132, 131, 132, 132, 132,
    130, 131, 131, 131, 130, 131, 131, 132, 130, 132, 131, 131, 130, 131, 131, 131,
    130, 132, 132, 131, 130, 131, 131, 132, 130, 132, 131, 131, 130, 131, 132, 132,
    129, 131, 131, 130, 129, 130, 130, 131, 129, 131, 130, 130, 129, 130, 131, 131,
    130, 132, 132, 131, 130, 131, 131, 132, 130, 132, 131, 131, 130, 131, 132, 132,
    131, 132, 132, 132, 131, 132, 131, 132, 131, 132, 131, 132, 131, 132, 132, 132,
    131, 132, 132, 132, 131, 132, 131, 132, 131, 132, 131, 132, 131, 132, 132, 132
  },
  {
    146, 147, 147, 146, 146, 147, 146, 147, 146, 147, 146, 147, 145, 146, 147, 147,
    147, 149, 149, 148, 147, 148, 148, 149, 147, 149, 148, 148, 147, 148, 149, 149,
    147, 149, 149, 148, 147, 148, 148, 149, 147, 149, 148, 148, 147, 148, 149, 149,
    146, 148, 148, 147, 146, 147, 147, 148, 146, 148, 147, 147, 146, 147, 148, 148,
    146, 147, 147, 146, 145, 147, 146, 147, 146, 147, 146, 147, 145, 146, 147, 147,
    147, 148, 148, 147, 147, 148, 147, 148, 147, 148, 147, 148, 147, 147, 148, 148,
    146, 148, 148, 147, 146, 147, 147, 148, 146, 148, 147, 147, 146, 147, 148, 148,
    147, 149, 149, 148, 147, 148, 148, 149, 147, 149, 148, 148, 147, 148, 149, 149,
    146, 147, 147, 146, 146, 147, 146, 147, 146, 147, 146, 147, 146, 146, 147, 147,
    147, 149, 149, 148, 147, 148, 148, 149, 147, 149, 148, 148, 147, 148, 149, 149,
    146, 148, 148, 147, 146, 147, 147, 148, 146, 148, 147, 147, 146, 147, 148, 148,
    147, 148, 148, 147, 147, 148, 147, 148, 147, 148, 147, 148, 147, 147, 148, 148,
    145, 147, 147, 146, 145, 147, 146, 147, 146, 147, 146, 147, 145, 146, 147, 147,
    146, 148, 148, 147, 146, 147, 147, 148, 146, 148, 147, 147, 146, 147, 148, 148,
    147, 149, 149, 148, 147, 148, 148, 149, 147, 149, 148, 148, 147, 148, 149, 149,
    147, 149, 149, 148, 147, 148, 148, 149, 147, 149, 148, 148, 147, 148, 149, 149
  }
};


Int g_aiDequantCoef64[6][64] =
{
  {
    20,  19, 25, 19, 20, 19, 25, 19,
    19,  18, 24, 18, 19, 18, 24, 18,
    25,  24, 32, 24, 25, 24, 32, 24,
    19,  18, 24, 18, 19, 18, 24, 18,
    20,  19, 25, 19, 20, 19, 25, 19,
    19,  18, 24, 18, 19, 18, 24, 18,
    25,  24, 32, 24, 25, 24, 32, 24,
    19,  18, 24, 18, 19, 18, 24, 18
  },
  {
    22,  21, 28, 21, 22, 21, 28, 21,
    21,  19, 26, 19, 21, 19, 26, 19,
    28,  26, 35, 26, 28, 26, 35, 26,
    21,  19, 26, 19, 21, 19, 26, 19,
    22,  21, 28, 21, 22, 21, 28, 21,
    21,  19, 26, 19, 21, 19, 26, 19,
    28,  26, 35, 26, 28, 26, 35, 26,
    21,  19, 26, 19, 21, 19, 26, 19
  },
  {
    26,  24, 33, 24, 26, 24, 33, 24,
    24,  23, 31, 23, 24, 23, 31, 23,
    33,  31, 42, 31, 33, 31, 42, 31,
    24,  23, 31, 23, 24, 23, 31, 23,
    26,  24, 33, 24, 26, 24, 33, 24,
    24,  23, 31, 23, 24, 23, 31, 23,
    33,  31, 42, 31, 33, 31, 42, 31,
    24,  23, 31, 23, 24, 23, 31, 23
  },
  {
    28,  26, 35, 26, 28, 26, 35, 26,
    26,  25, 33, 25, 26, 25, 33, 25,
    35,  33, 45, 33, 35, 33, 45, 33,
    26,  25, 33, 25, 26, 25, 33, 25,
    28,  26, 35, 26, 28, 26, 35, 26,
    26,  25, 33, 25, 26, 25, 33, 25,
    35,  33, 45, 33, 35, 33, 45, 33,
    26,  25, 33, 25, 26, 25, 33, 25
  },
  {
    32,  30, 40, 30, 32, 30, 40, 30,
    30,  28, 38, 28, 30, 28, 38, 28,
    40,  38, 51, 38, 40, 38, 51, 38,
    30,  28, 38, 28, 30, 28, 38, 28,
    32,  30, 40, 30, 32, 30, 40, 30,
    30,  28, 38, 28, 30, 28, 38, 28,
    40,  38, 51, 38, 40, 38, 51, 38,
    30,  28, 38, 28, 30, 28, 38, 28
  },
  {
    36,  34, 46, 34, 36, 34, 46, 34,
    34,  32, 43, 32, 34, 32, 43, 32,
    46,  43, 58, 43, 46, 43, 58, 43,
    34,  32, 43, 32, 34, 32, 43, 32,
    36,  34, 46, 34, 36, 34, 46, 34,
    34,  32, 43, 32, 34, 32, 43, 32,
    46,  43, 58, 43, 46, 43, 58, 43,
    34,  32, 43, 32, 34, 32, 43, 32
  }
  
};

UInt g_aiQuantCoef64[6][64] =
{
  {
    0x3333, 0x2fbe, 0x4189, 0x2fbe, 0x3333, 0x2fbe, 0x4189, 0x2fbe,
    0x2fbe, 0x2ca4, 0x3c79, 0x2ca4, 0x2fbe, 0x2ca4, 0x3c79, 0x2ca4,
    0x4189, 0x3c79, 0x51ec, 0x3c79, 0x4189, 0x3c79, 0x51ec, 0x3c79,
    0x2fbe, 0x2ca4, 0x3c79, 0x2ca4, 0x2fbe, 0x2ca4, 0x3c79, 0x2ca4,
    0x3333, 0x2fbe, 0x4189, 0x2fbe, 0x3333, 0x2fbe, 0x4189, 0x2fbe,
    0x2fbe, 0x2ca4, 0x3c79, 0x2ca4, 0x2fbe, 0x2ca4, 0x3c79, 0x2ca4,
    0x4189, 0x3c79, 0x51ec, 0x3c79, 0x4189, 0x3c79, 0x51ec, 0x3c79,
    0x2fbe, 0x2ca4, 0x3c79, 0x2ca4, 0x2fbe, 0x2ca4, 0x3c79, 0x2ca4,
  },
  {
    0x2e8c, 0x2b32, 0x3a84, 0x2b32, 0x2e8c, 0x2b32, 0x3a84, 0x2b32,
    0x2b32, 0x2a4a, 0x37d2, 0x2a4a, 0x2b32, 0x2a4a, 0x37d2, 0x2a4a,
    0x3a84, 0x37d2, 0x4ae6, 0x37d2, 0x3a84, 0x37d2, 0x4ae6, 0x37d2,
    0x2b32, 0x2a4a, 0x37d2, 0x2a4a, 0x2b32, 0x2a4a, 0x37d2, 0x2a4a,
    0x2e8c, 0x2b32, 0x3a84, 0x2b32, 0x2e8c, 0x2b32, 0x3a84, 0x2b32,
    0x2b32, 0x2a4a, 0x37d2, 0x2a4a, 0x2b32, 0x2a4a, 0x37d2, 0x2a4a,
    0x3a84, 0x37d2, 0x4ae6, 0x37d2, 0x3a84, 0x37d2, 0x4ae6, 0x37d2,
    0x2b32, 0x2a4a, 0x37d2, 0x2a4a, 0x2b32, 0x2a4a, 0x37d2, 0x2a4a,
  },
  {
    0x2762, 0x25cb, 0x31a6, 0x25cb, 0x2762, 0x25cb, 0x31a6, 0x25cb,
    0x25cb, 0x22ef, 0x2ed1, 0x22ef, 0x25cb, 0x22ef, 0x2ed1, 0x22ef,
    0x31a6, 0x2ed1, 0x3e6a, 0x2ed1, 0x31a6, 0x2ed1, 0x3e6a, 0x2ed1,
    0x25cb, 0x22ef, 0x2ed1, 0x22ef, 0x25cb, 0x22ef, 0x2ed1, 0x22ef,
    0x2762, 0x25cb, 0x31a6, 0x25cb, 0x2762, 0x25cb, 0x31a6, 0x25cb,
    0x25cb, 0x22ef, 0x2ed1, 0x22ef, 0x25cb, 0x22ef, 0x2ed1, 0x22ef,
    0x31a6, 0x2ed1, 0x3e6a, 0x2ed1, 0x31a6, 0x2ed1, 0x3e6a, 0x2ed1,
    0x25cb, 0x22ef, 0x2ed1, 0x22ef, 0x25cb, 0x22ef, 0x2ed1, 0x22ef,
  },
  {
    0x2492, 0x22e3, 0x2ed0, 0x22e3, 0x2492, 0x22e3, 0x2ed0, 0x22e3,
    0x22e3, 0x2024, 0x2bfb, 0x2024, 0x22e3, 0x2024, 0x2bfb, 0x2024,
    0x2ed0, 0x2bfb, 0x3a41, 0x2bfb, 0x2ed0, 0x2bfb, 0x3a41, 0x2bfb,
    0x22e3, 0x2024, 0x2bfb, 0x2024, 0x22e3, 0x2024, 0x2bfb, 0x2024,
    0x2492, 0x22e3, 0x2ed0, 0x22e3, 0x2492, 0x22e3, 0x2ed0, 0x22e3,
    0x22e3, 0x2024, 0x2bfb, 0x2024, 0x22e3, 0x2024, 0x2bfb, 0x2024,
    0x2ed0, 0x2bfb, 0x3a41, 0x2bfb, 0x2ed0, 0x2bfb, 0x3a41, 0x2bfb,
    0x22e3, 0x2024, 0x2bfb, 0x2024, 0x22e3, 0x2024, 0x2bfb, 0x2024,
  },
  {
    0x2000, 0x1e3c, 0x28f6, 0x1e3c, 0x2000, 0x1e3c, 0x28f6, 0x1e3c,
    0x1e3c, 0x1cb2, 0x2631, 0x1cb2, 0x1e3c, 0x1cb2, 0x2631, 0x1cb2,
    0x28f6, 0x2631, 0x3367, 0x2631, 0x28f6, 0x2631, 0x3367, 0x2631,
    0x1e3c, 0x1cb2, 0x2631, 0x1cb2, 0x1e3c, 0x1cb2, 0x2631, 0x1cb2,
    0x2000, 0x1e3c, 0x28f6, 0x1e3c, 0x2000, 0x1e3c, 0x28f6, 0x1e3c,
    0x1e3c, 0x1cb2, 0x2631, 0x1cb2, 0x1e3c, 0x1cb2, 0x2631, 0x1cb2,
    0x28f6, 0x2631, 0x3367, 0x2631, 0x28f6, 0x2631, 0x3367, 0x2631,
    0x1e3c, 0x1cb2, 0x2631, 0x1cb2, 0x1e3c, 0x1cb2, 0x2631, 0x1cb2,
  },
  {
    0x1c72, 0x1aae, 0x239e, 0x1aae, 0x1c72, 0x1aae, 0x239e, 0x1aae,
    0x1aae, 0x191c, 0x21c0, 0x191c, 0x1aae, 0x191c, 0x21c0, 0x191c,
    0x239e, 0x21c0, 0x2d32, 0x21c0, 0x239e, 0x21c0, 0x2d32, 0x21c0,
    0x1aae, 0x191c, 0x21c0, 0x191c, 0x1aae, 0x191c, 0x21c0, 0x191c,
    0x1c72, 0x1aae, 0x239e, 0x1aae, 0x1c72, 0x1aae, 0x239e, 0x1aae,
    0x1aae, 0x191c, 0x21c0, 0x191c, 0x1aae, 0x191c, 0x21c0, 0x191c,
    0x239e, 0x21c0, 0x2d32, 0x21c0, 0x239e, 0x21c0, 0x2d32, 0x21c0,
    0x1aae, 0x191c, 0x21c0, 0x191c, 0x1aae, 0x191c, 0x21c0, 0x191c,
  }
};
#endif //E243_CORE_TRANSFORMS
const UChar g_aucChromaScale[52]=
{
  0, 1, 2, 3, 4, 5, 6, 7, 8, 9,10,11,
  12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,
  28,29,29,30,31,32,32,33,34,34,35,35,36,36,37,37,
  37,38,38,38,39,39,39,39
};
// ====================================================================================================================
// TENTM VLC table
// ====================================================================================================================

#if RUNLEVEL_TABLE_CUT || CAVLC_RQT_CBP
#define M1 MAX_UINT
#else
#define M1 (UInt)-1
#endif

// Below table need to be optimized
const UInt g_auiCbpVlcNum[2][8] =
{
  {0, 0, 0, 0, 0, 0, 0, 0},
  {0, 0, 0, 0, 0, 0, 0, 0}
};

#if CAVLC_RQT_CBP
const UInt g_auiCBP_YUV_TableE[4][8] = 
{
  {2,5,6,7,0,3,4,1},
  {0,4,6,7,1,2,3,5},
  {2,5,6,7,0,3,4,1},
  {0,4,6,7,1,2,3,5}
};
const UInt g_auiCBP_YUV_TableD[4][8] = 
{
  {4,7,0,5,6,1,2,3},
  {0,4,5,6,1,7,2,3},
  {4,7,0,5,6,1,2,3},
  {0,4,5,6,1,7,2,3}
};
const UInt g_auiCBP_YS_TableE[2][4] = 
{
  {3,  2, 0, 1},
  {2, M1, 0, 1}
};

const UInt g_auiCBP_YS_TableD[2][4] = 
{
  {  2,  3,  1,  0},
  {  2,  3,  0, M1}
};

const UInt g_auiCBP_YC_TableE[2][4] =
{ 
  {  2,  1,  3,  0}, 
  {  0,  2,  1,  3}
};

const UInt g_auiCBP_YC_TableD[2][4] =
{ 
  {3,1,0,2}, 
  {0,2,1,3}
};

const UInt g_auiCBP_YCS_Table[2][8] = 
{
  {  0,  3,  9, 11,  8, 20, 42, 43},
  {  1,  1,  1,  1,  1,  1,  0, M1}
};

const UInt g_auiCBP_YCS_TableLen[2][8] = 
{
  {  1,  2,  4,  4,  4,  5,  6,  6},
  {  1,  2,  3,  4,  5,  6,  6,  0}
};

const UInt g_auiCBP_YCS_TableE[2][8] = 
{
  {4,  5,  6,  7,  1,  2,  0,  3},
  {0,  7,  5,  6,  2,  1,  4,  3}
};

const UInt g_auiCBP_YCS_TableD[2][8] = 
{
  {6,  4,  5,  7,  0,  1,  2,  3},
  {0,  5,  4,  7,  6,  2,  3, 1}
};

const UInt g_auiCBP_4Y_TableE[2][15] = 
{
  {14, 13, 10, 12,  9,  8,  4, 11,  7,  6,  3,  5,  2,  1,  0},
  { 0,  1,  4,  2,  5,  6, 10,  3,  7,  8, 11,  9, 12, 13, 14}
};

const UInt g_auiCBP_4Y_TableD[2][15] = 
{
  {14, 13, 12, 10, 6, 11, 9, 8, 5, 4, 2, 7, 3, 1, 0},
  {0, 1, 3, 7, 2, 4, 5, 8, 9, 11, 6, 10 ,12, 13, 14} 
};

const UInt g_auiCBP_4Y_VlcNum[15] = 
{
  1,  2,  2,  2, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11
};
#else
const UInt g_auiCBPTableE[2][8] = 
{
  {2,0,6,4,5,3,7,1},
  {0,1,6,3,4,2,7,5}
};

const UInt g_auiCBPTableD[2][8] = 
{
  {1,7,0,5,3,4,2,6},
  {0,1,5,3,4,7,2,6}
};

const UInt g_auiBlkCBPTableE[2][15] = 
{
  {14, 13, 10, 12, 9, 8, 4, 11, 7, 6, 3, 5, 2, 1, 0},
  {0, 1, 4, 2, 5, 6, 10, 3, 7, 8, 11, 9, 12, 13, 14}
};

const UInt g_auiBlkCBPTableD[2][15] = 
{
  {14, 13, 12, 10, 6, 11, 9, 8, 5, 4, 2, 7, 3, 1, 0},
  {0, 1, 3, 7, 2, 4, 5, 8, 9, 11, 6, 10 ,12, 13, 14} 
};

const UInt g_auiBlkCbpVlcNum[15] = 
{
  1,  2,  2,  2, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11
};
#endif

#if UNIFY_INTER_TABLE
const UInt g_auiComMI1TableE[9] = {0,1,2,3,4,5,6,7,8};
const UInt g_auiComMI1TableD[9] = {0,1,2,3,4,5,6,7,8};
#else
const UInt g_auiMI1TableE[8] = {0,2,1,4,3,6,5,7};
const UInt g_auiMI1TableD[8] = {0,2,1,4,3,6,5,7};
const UInt g_auiMI2TableE[15] = {0,1,3,2,6,5,4,7,9,8,13,12,11,14,10};
const UInt g_auiMI2TableD[15] = {0,1,3,2,6,5,4,7,9,8,14,12,11,10,13};

const UInt g_auiMI1TableENoL1[8] = {0,1,6,7,2,4,3,5};
const UInt g_auiMI1TableDNoL1[8] = {0,1,4,6,5,7,2,3};
const UInt g_auiMI2TableENoL1[15] = {0,1,2,12,14,13,3,4,6,5,10,9,8,11,7};
const UInt g_auiMI2TableDNoL1[15] = {0,1,2,6,7,9,8,14,12,11,10,13,3,5,4};

#if MS_LCEC_ONE_FRAME
// as there are two lists used for bi-directional prediction, and one list for uni-directional prediction
// for uni-directional prediction, when pcSlice->getNoBackPredFlag() is true, list 0 is used for uni-directional prediction
// when pcSlice->getNoBackPredFlag() is false, combined list is used for uni-directional prediction
// the meaning of the elements are as follows:
/*
 0: uni-directional prediction, frame 0
 1: uni-directional prediction, frame 1
 2: uni-directional prediction, frame 2
 3: uni-directional prediction, frame 3
 4: bi-directional prediction, frame (0, 0)
 5: bi-directional prediction, frame (0, 1)
 6: bi-directional prediction, frame (1, 0)
 7: bi-directional prediction, frame (1, 1)
 8: exception, for uni-directional prediction, if combined list, reference frame index >= 4 (count from 0) is exception
 if GPB (only list0 is used), reference frame index >= 2 (count from 0) is exception
 for bi-directional prediction, reference frame index >= 2 (count from 0 ) is exception
 */
const UInt g_auiMI1TableEOnly1Ref[8] = {0,3,1,4,2,6,5,7};
const UInt g_auiMI1TableDOnly1Ref[8] = {0,2,4,1,3,6,5,7};
const UInt g_auiMI1TableEOnly1RefNoL1[8] = {0,2,3,4,1,6,5,7};
const UInt g_auiMI1TableDOnly1RefNoL1[8] = {0,4,1,2,3,6,5,7};
#endif
#endif

#if QC_LCEC_INTER_MODE
const UInt g_auiInterModeTableE[4][7] = {{0,1,2,3,4,5,6},{0,1,2,3,4,5,6},{0,1,2,3,4,5,6},{6,0,1,2,3,4,5}};
const UInt g_auiInterModeTableD[4][7] = {{0,1,2,3,4,5,6},{0,1,2,3,4,5,6},{0,1,2,3,4,5,6},{1,2,3,4,5,6,0}};
#endif



// Below table need to be optimized
const UInt g_auiMITableVlcNum[15] = 
{
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
};


const UInt g_auiLPTableE4[3][32] =
{
  {0,1,2,3,5,4,7,6,9,11,14,8,16,15,10,13,12,17,18,19,25,23,20,22,28,26,29,24,30,31,27,21},  //4x4I
  {0,1,2,7,5,3,6,4,11,8,12,10,9,14,13,15,16,17,21,27,26,18,19,23,29,20,25,28,22,30,24,31},  //4x4P
  {0,1,2,7,5,3,6,4,11,8,12,10,9,14,13,15,16,17,21,27,26,18,19,23,29,20,25,28,22,30,24,31}   //4x4B
};

const UInt g_auiLPTableD4[3][32] =
{
  {0,1,2,3,5,4,7,6,11,8,14,9,16,15,10,13,12,17,18,19,22,31,23,21,27,20,25,30,24,26,28,29},  //4x4I
  {0,1,2,5,7,4,6,3,9,12,11,8,10,14,13,15,16,17,21,22,25,18,28,23,30,26,20,19,27,24,29,31},  //4x4P
  {0,1,2,5,7,4,6,3,9,12,11,8,10,14,13,15,16,17,21,22,25,18,28,23,30,26,20,19,27,24,29,31}   //4x4B
};

#if !CAVLC_COEF_LRG_BLK
const UInt g_auiLPTableE8[8][128] =
{
  {0,2,1,4,7,3,6,11,13,9,18,22,19,17,8,10,20,24,28,27,21,26,38,42,37,31,23,12,16,33,30,44,45,49,46,36,57,50,55,47,39,43,35,75,66,56,60,58,54,70,63,29,77,87,69,48,51,64,71,65,80,85,76,94,5,15,14,34,52,25,41,72,74,53,59,81,82,68,40,62,95,127,89,84,67,73,126,79,125,93,90,61,32,124,123,83,86,122,88,78,121,92,91,120,119,118,117,116,115,114,113,112,111,110,109,108,107,106,105,104,103,102,101,100,99,98,97,96}, //U
  {0,2,1,4,7,3,6,11,13,9,18,22,19,17,8,10,20,24,28,27,21,26,38,42,37,31,23,12,16,33,30,44,45,49,46,36,57,50,55,47,39,43,35,75,66,56,60,58,54,70,63,29,77,87,69,48,51,64,71,65,80,85,76,94,5,15,14,34,52,25,41,72,74,53,59,81,82,68,40,62,95,127,89,84,67,73,126,79,125,93,90,61,32,124,123,83,86,122,88,78,121,92,91,120,119,118,117,116,115,114,113,112,111,110,109,108,107,106,105,104,103,102,101,100,99,98,97,96}, //V
  
  {0,1,2,6,7,3,5,12,11,10,8,17,20,22,9,13,32,21,14,25,16,29,62,59,42,28,15,18,19,26,35,36,30,48,56,39,91,72,63,37,38,44,23,27,41,47,40,58,77,83,52,65,57,33,53,60,54,69,89,45,34,51,46,49,4,31,24,71,88,70,92,103,119,93,55,95,122,127,106,123,125,76,66,126,108,112,117,107,124,111,68,97,85,114,113,121,116,115,79,82,105,120,110,118,109,102,100,84,101,90,104,94,96,78,86,99,81,75,87,98,80,73,74,67,64,61,43,50}, //8x8I
  {0,1,2,7,5,3,11,16,6,8,14,18,13,19,9,15,21,22,24,25,20,31,44,36,37,30,23,10,4,26,28,33,39,45,49,40,55,56,59,43,42,32,12,17,38,48,50,60,74,79,75,47,51,27,35,53,65,73,76,66,46,80,77,83,29,71,69,88,92,81,90,106,97,86,85,116,127,115,93,101,114,126,113,102,87,84,104,125,124,123,122,82,34,103,121,120,107,112,94,41,68,105,111,119,110,95,61,57,96,109,118,108,52,54,100,117,99,62,78,98,91,58,64,89,67,72,63,70}, //8x8P
  {0,1,2,7,5,3,11,16,6,8,14,18,13,19,9,15,21,22,24,25,20,31,44,36,37,30,23,10,4,26,28,33,39,45,49,40,55,56,59,43,42,32,12,17,38,48,50,60,74,79,75,47,51,27,35,53,65,73,76,66,46,80,77,83,29,71,69,88,92,81,90,106,97,86,85,116,127,115,93,101,114,126,113,102,87,84,104,125,124,123,122,82,34,103,121,120,107,112,94,41,68,105,111,119,110,95,61,57,96,109,118,108,52,54,100,117,99,62,78,98,91,58,64,89,67,72,63,70}, //8x8B
  
  {0,1,2,5,4,3,6,7,8,9,10,11,13,12,15,16,20,17,14,18,23,29,26,19,24,38,22,34,32,46,21,39,41,30,28,45,27,40,56,59,35,42,51,48,44,43,47,54,58,36,53,63,61,37,55,65,50,62,60,52,57,49,31,25,33,64,67,69,72,70,80,86,95,99,89,123,88,96,94,113,101,91,98,93,83,108,110,107,103,105,79,77,90,85,127,81,114,104,118,84,97,102,122,87,106,109,92,117,126,78,116,71,112,120,82,121,75,66,111,124,76,100,125,68,73,115,119,74}, //16x16I
  {0,2,1,4,5,3,6,7,8,11,14,13,12,10,9,15,17,16,20,23,22,26,31,30,27,21,25,19,24,32,28,29,36,35,40,37,47,42,44,38,33,34,41,43,45,39,48,49,54,59,60,57,52,51,56,58,61,55,65,62,63,64,67,69,18,50,46,72,66,53,70,73,71,74,89,81,79,77,68,83,78,84,80,87,90,96,127,93,95,86,94,75,82,99,107,126,92,100,88,76,85,106,125,124,98,97,105,104,123,122,121,103,91,102,120,119,118,117,101,116,115,114,113,112,111,110,109,108}, //16x16P
  {0,2,1,4,5,3,6,7,8,11,14,13,12,10,9,15,17,16,20,23,22,26,31,30,27,21,25,19,24,32,28,29,36,35,40,37,47,42,44,38,33,34,41,43,45,39,48,49,54,59,60,57,52,51,56,58,61,55,65,62,63,64,67,69,18,50,46,72,66,53,70,73,71,74,89,81,79,77,68,83,78,84,80,87,90,96,127,93,95,86,94,75,82,99,107,126,92,100,88,76,85,106,125,124,98,97,105,104,123,122,121,103,91,102,120,119,118,117,101,116,115,114,113,112,111,110,109,108}  //16x16B 
};

const UInt g_auiLPTableD8[8][128] =
{
  {0,2,1,5,3,64,6,4,14,9,15,7,27,8,66,65,28,13,10,12,16,20,11,26,17,69,21,19,18,51,30,25,92,29,67,42,35,24,22,40,78,70,23,41,31,32,34,39,55,33,37,56,68,73,48,38,45,36,47,74,46,91,79,50,57,59,44,84,77,54,49,58,71,85,72,43,62,52,99,87,60,75,76,95,83,61,96,53,98,82,90,102,101,89,63,80,127,126,125,124,123,122,121,120,119,118,117,116,115,114,113,112,111,110,109,108,107,106,105,104,103,100,97,94,93,88,86,81}, //U
  {0,2,1,5,3,64,6,4,14,9,15,7,27,8,66,65,28,13,10,12,16,20,11,26,17,69,21,19,18,51,30,25,92,29,67,42,35,24,22,40,78,70,23,41,31,32,34,39,55,33,37,56,68,73,48,38,45,36,47,74,46,91,79,50,57,59,44,84,77,54,49,58,71,85,72,43,62,52,99,87,60,75,76,95,83,61,96,53,98,82,90,102,101,89,63,80,127,126,125,124,123,122,121,120,119,118,117,116,115,114,113,112,111,110,109,108,107,106,105,104,103,100,97,94,93,88,86,81}, //V 
  
  {0,1,2,5,64,6,3,4,10,14,9,8,7,15,18,26,20,11,27,28,12,17,13,42,66,19,29,43,25,21,32,65,16,53,60,30,31,39,40,35,46,44,24,126,41,59,62,45,33,63,127,61,50,54,56,74,34,52,47,23,55,125,22,38,124,51,82,123,90,57,69,67,37,121,122,117,81,48,113,98,120,116,99,49,107,92,114,118,68,58,109,36,70,73,111,75,112,91,119,115,106,108,105,71,110,100,78,87,84,104,102,89,85,94,93,97,96,86,103,72,101,95,76,79,88,80,83,77}, //8x8I
  {0,1,2,5,28,4,8,3,9,14,27,6,42,12,10,15,7,43,11,13,20,16,17,26,18,19,29,53,30,64,25,21,41,31,92,54,23,24,44,32,35,99,40,39,22,33,60,51,45,34,46,52,112,55,113,36,37,107,121,38,47,106,117,126,122,56,59,124,100,66,127,65,125,57,48,50,58,62,118,49,61,69,91,63,85,74,73,84,67,123,70,120,68,78,98,105,108,72,119,116,114,79,83,93,86,101,71,96,111,109,104,102,97,82,80,77,75,115,110,103,95,94,90,89,88,87,81,76}, //8x8P
  {0,1,2,5,28,4,8,3,9,14,27,6,42,12,10,15,7,43,11,13,20,16,17,26,18,19,29,53,30,64,25,21,41,31,92,54,23,24,44,32,35,99,40,39,22,33,60,51,45,34,46,52,112,55,113,36,37,107,121,38,47,106,117,126,122,56,59,124,100,66,127,65,125,57,48,50,58,62,118,49,61,69,91,63,85,74,73,84,67,123,70,120,68,78,98,105,108,72,119,116,114,79,83,93,86,101,71,96,111,109,104,102,97,82,80,77,75,115,110,103,95,94,90,89,88,87,81,76}, //8x8B
  
  {0,1,2,5,4,3,6,7,8,9,10,11,13,12,18,14,15,17,19,23,16,30,26,20,24,63,22,36,34,21,33,62,28,64,27,40,49,53,25,31,37,32,41,45,44,35,29,46,43,61,56,42,59,50,47,54,38,60,48,39,58,52,57,51,65,55,117,66,123,67,69,111,68,124,127,116,120,91,109,90,70,95,114,84,99,93,71,103,76,74,92,81,106,83,78,72,77,100,82,73,121,80,101,88,97,89,104,87,85,105,86,118,112,79,96,125,110,107,98,126,113,115,102,75,119,122,108,94}, //16x16I
  {0,2,1,5,3,4,6,7,8,14,13,9,12,11,10,15,17,16,64,27,18,25,20,19,28,26,21,24,30,31,23,22,29,40,41,33,32,35,39,45,34,42,37,43,38,44,66,36,46,47,65,53,52,69,48,57,54,51,55,49,50,56,59,60,61,58,68,62,78,63,70,72,67,71,73,91,99,77,80,76,82,75,92,79,81,100,89,83,98,74,84,112,96,87,90,88,85,105,104,93,97,118,113,111,107,106,101,94,127,126,125,124,123,122,121,120,119,117,116,115,114,110,109,108,103,102,95,86}, //16x16P
  {0,2,1,5,3,4,6,7,8,14,13,9,12,11,10,15,17,16,64,27,18,25,20,19,28,26,21,24,30,31,23,22,29,40,41,33,32,35,39,45,34,42,37,43,38,44,66,36,46,47,65,53,52,69,48,57,54,51,55,49,50,56,59,60,61,58,68,62,78,63,70,72,67,71,73,91,99,77,80,76,82,75,92,79,81,100,89,83,98,74,84,112,96,87,90,88,85,105,104,93,97,118,113,111,107,106,101,94,127,126,125,124,123,122,121,120,119,117,116,115,114,110,109,108,103,102,95,86}  //16x16B  
};
#endif

#if LCEC_INTRA_MODE
#if MTK_DCM_MPM

const UInt  g_auiIntraModeTableD17[2][16]=
{
  {0, 15, 11, 10, 13,  7,  9, 4, 14,  2, 3, 6,  8, 5, 12, 1},
  {14,10,  9,  0, 13,  7,  2, 8,  3, 12, 6, 4, 11, 1,  5, 0}  
};
const UInt  g_auiIntraModeTableE17[2][16]=
{
  {0, 15,  9, 10,  7, 13, 11, 5, 12, 6, 3,  2, 14, 4,  8, 1},
  {3, 13,  6,  8, 11, 14, 10, 5,  7, 2, 1, 12,  9, 4,  0, 0}
};
const UInt  g_auiIntraModeTableD34[2][33]=
{
  {2, 0, 29, 30, 20,  1, 21, 28, 15,  7, 16,  8, 11, 31, 22, 19, 32, 10, 27, 14,  4, 18, 12, 23, 17, 24, 26,  6, 25,  5, 9, 3, 13},
  {2, 1, 28,  0, 29, 20, 27, 19, 15, 21,  7, 14, 10, 11, 30, 31, 18, 16, 26,  8, 17,  4, 22,  9, 23,  6, 25, 13, 24, 12, 3, 5,  0}
};
const UInt  g_auiIntraModeTableE34[2][33]=
{
  {1, 5, 0, 31, 20, 29, 27,  9, 11, 30, 17, 12, 22, 32, 19, 8, 10, 24, 21, 15, 4, 6, 14, 23, 25, 28, 26, 18, 7, 2,  3, 13, 16},
  {3, 1, 0, 30, 21, 31, 25, 10, 19, 23, 12, 13, 29, 27, 11, 8, 17, 20, 16,  7, 5, 9, 22, 24, 28, 26, 18,  6, 2, 4, 14, 15,  0}
};

#else
const UInt  g_auiIntraModeTableD17[16]={15,0,11,7,14,10,13,4,9,2,3,1,8,6,12,5};
const UInt  g_auiIntraModeTableE17[16]={1,11,9,10,7,15,13,3,12,8,5,2,14,6,4,0};

const UInt  g_auiIntraModeTableD34[33]={0,29,1,2,30,20,28,15,32,21,27,11,10,7,19,14,16,23,22,8,4,18,31,26,17,9,25,6,3,12,13,24,5};
const UInt  g_auiIntraModeTableE34[33]={0,2,3,28,20,32,27,13,19,25,12,11,29,30,15,7,16,24,21,14,5,9,18,17,31,26,23,10,6,1,4,22,8};
#endif
#endif

const UInt g_auiLastPosVlcIndex[10] = {0,0,0,0,0,0,0,0,0,0};

const UInt g_auiLastPosVlcNum[10][17] =
{
  {10,10,10,10, 2,2,2,7,9,9,9,9,9,4,4,4,4},
  {10,10,10,10,10,2,9,9,9,9,9,9,9,4,4,4,4},
#if CAVLC_COEF_LRG_BLK 
  { 2, 2, 2, 2, 2, 7, 7, 7, 7, 7, 7, 7, 7, 7, 4, 4,13},
  { 2, 2, 2, 2, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,13},
  { 2, 2, 2, 2, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,13},
  {10,10,10, 4, 4, 4, 4,12,12,12,12,12,12,12,12,12,12},
  {10,10,10,10, 4, 4,12,12,12,12,12,12,12,12,12,12,12},
  {10,10,10,10, 4, 4,12,12,12,12,12,12,12,12,12,12,12},
#else
  { 2, 2, 2, 2, 2,7,7,7,7,7,7,7,7,7,4,4,4},
  { 2, 2, 2, 2, 7,7,7,7,7,7,7,7,7,7,7,7,4},
  { 2, 2, 2, 2, 7,7,7,7,7,7,7,7,7,7,7,7,4},
  {10, 1, 2, 2, 2,2,7,7,7,7,9,9,9,4,4,4,4},
  {10,10, 2, 2, 7,7,7,7,7,7,7,7,4,4,4,4,4},
  {10,10, 2, 2, 7,7,7,7,7,7,7,7,4,4,4,4,4},
#endif
  { 2, 2, 2, 2, 7,7,7,7,7,7,7,7,7,7,7,7,4},
  { 2, 2, 2, 2, 7,7,7,7,7,7,7,7,7,7,7,7,4}
};



#if RUNLEVEL_TABLE_CUT
const UInt g_auiLumaRun8x8[28][29] =
{
  {1,   0,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1},
  {2,   1,   0,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1},
  {1,   3,   2,   0,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1},
  {2,   1,   3,   4,   0,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1},
  {1,   5,   3,   2,   4,   0,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1},
  {1,   2,   6,   5,   3,   4,   0,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1},
  {2,   1,   3,   5,   4,   7,   6,   0,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1},
  {1,   5,   4,   2,   3,   6,   8,   7,   0,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1},
  {1,   3,   8,   7,   5,   2,   4,   9,   6,   0,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1},
  {1,   2,   5,   9,   8,   7,   3,   4,  10,   6,   0,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1},
  {2,   1,   3,   4,   7,   8,   5,   6,   9,  11,  10,   0,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1},
  {1,   4,   3,   2,   5,   7,   6,   8,  10,  11,  12,   9,   0,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1},
  {1,   2,   6,   7,   5,   3,   4,   8,   9,  13,  11,  12,  10,   0,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1},
  {1,   2,   4,   8,   9,   7,   6,   3,   5,  12,  14,  13,  11,  10,   0,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1},
  {1,   2,   3,   5,   9,  10,   8,   7,   4,   6,  12,  15,  14,  13,  11,   0,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1},
  {1,   2,   3,   4,   5,   8,  10,   7,   6,   9,  11,  13,  16,  14,  15,  12,   0,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1},
  {1,   3,   4,   2,   5,   6,  10,   9,   7,   8,  11,  12,  13,  15,  16,  17,  14,   0,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1},
  {1,   2,   4,   7,   6,   3,   5,   8,   9,  10,  11,  17,  15,  13,  14,  16,  18,  12,   0,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1},
  {1,   2,   5,   7,   9,  10,   6,   3,   4,   8,  11,  16,  19,  17,  15,  13,  14,  18,  12,   0,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1},
  {1,   3,   5,   7,   9,  14,  12,   8,   6,   2,   4,  15,  17,  19,  18,  16,  10,  11,  20,  13,   0,  M1,  M1,  M1,  M1,  M1,  M1,  M1,  M1},
  {1,   2,   4,   6,   9,   8,  15,  14,  13,   7,   3,   5,  16,  18,  21,  19,  17,  10,  12,  20,  11,   0,  M1,  M1,  M1,  M1,  M1,  M1,  M1},
  {2,   1,   3,   4,   5,   8,   9,  11,  13,  10,   6,   7,  12,  14,  19,  22,  21,  17,  15,  18,  20,  16,   0,  M1,  M1,  M1,  M1,  M1,  M1},
  {1,   5,   4,   2,   3,   6,   8,   7,  11,  12,   9,  10,  13,  14,  15,  18,  22,  21,  17,  19,  20,  23,  16,   0,  M1,  M1,  M1,  M1,  M1},
  {1,   4,   8,   6,   5,   2,   3,   7,   9,  12,  10,  11,  13,  16,  15,  14,  18,  20,  21,  19,  23,  22,  24,  17,   0,  M1,  M1,  M1,  M1},
  {1,   3,   7,  12,  10,   6,   5,   2,   4,   8,   9,  11,  13,  16,  23,  20,  15,  14,  18,  19,  21,  24,  25,  22,  17,   0,  M1,  M1,  M1},
  {1,   2,   5,   9,  13,  11,  10,   8,   6,   3,   4,   7,  12,  15,  18,  25,  22,  19,  16,  14,  17,  21,  26,  23,  24,  20,   0,  M1,  M1},
  {1,   2,   4,   7,  11,  15,  12,   9,  10,   8,   5,   3,   6,  14,  16,  22,  23,  24,  20,  19,  13,  18,  26,  27,  25,  21,  17,   0,  M1},
  {1,   2,   3,   5,   6,  10,  14,  13,  11,   9,  12,   7,   4,   8,  16,  18,  21,  23,  25,  20,  22,  15,  19,  27,  28,  26,  24,  17,   0}
};
#else
const UInt g_auiLumaRun8x8[29][2][64] =
{
  /* 0 */
  {
    {
      1,  0, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    },
    {
      2, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    }
  },
  /* 1 */
  {
    {
      2,  1,  0, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    },
    {
      4,  3, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    }
  },
  /* 2 */
  {
    {
      1,  3,  2,  0, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    },
    {
      4,  6,  5, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    }
  },
  /* 3 */
  {
    {
      2,  1,  3,  4,  0, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    },
    {
      6,  5,  7,  8, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    }
  },
  /* 4 */
  {
    {
      1,  5,  3,  2,  4,  0, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    },
    {
      6, 10,  8,  7,  9, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    }
  },
  /* 5 */
  {
    {
      1,  2,  6,  5,  3,  4,  0, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    },
    {
      7,  8, 11, 10,  9, 12, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    }
  },
  /* 6 */
  {
    {
      2,  1,  3,  5,  4,  7,  6,  0, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    },
    {
      9,  8, 10, 12, 11, 13, 14, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    }
  },
  /* 7 */
  {
    {
      1,  5,  4,  2,  3,  6,  8,  7,  0, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    },
    {
      9, 13, 12, 10, 11, 14, 15, 16, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    }
  },
  /* 8 */
  {
    {
      1,  3,  8,  7,  5,  2,  4,  9,  6,  0, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    },
    {
      10, 12, 16, 15, 14, 11, 13, 18, 17, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    }
  },
  /* 9 */
  {
    {
      1,  2,  5, 10,  9,  7,  3,  4, 11,  6,  0, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    },
    {
      8, 12, 15, 17, 18, 16, 13, 14, 20, 19, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    }
  },
  {
    {
      2,  1,  3,  4,  7,  8,  5,  6,  9, 11, 10,  0, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    },
    {
      13, 12, 14, 15, 17, 18, 16, 19, 20, 22, 21, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    }
  },
  {
    {
      1,  4,  3,  2,  5,  7,  6,  8, 10, 11, 12,  9,  0, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    },
    {
      13, 17, 15, 14, 16, 19, 18, 20, 22, 23, 24, 21, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    }
  },
  {
    {
      1,  2,  6,  7,  5,  3,  4,  8,  9, 13, 11, 12, 10,  0, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    },
    {
      14, 16, 18, 21, 19, 15, 17, 20, 22, 26, 23, 24, 25, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    }
  },
  {
    {
      1,  2,  4,  8,  9,  7,  6,  3,  5, 12, 14, 13, 11, 10,  0, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    },
    {
      15, 16, 19, 21, 23, 22, 20, 17, 18, 26, 28, 27, 24, 25, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    }
  },
  {
    {
      1,  2,  3,  5,  9, 10,  8,  7,  4,  6, 13, 16, 15, 14, 12,  0,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    },
    {
      11, 17, 18, 20, 22, 24, 23, 26, 19, 21, 27, 30, 29, 25, 28, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    }
  },
  {
    {
      1,  2,  3,  4,  5,  8, 10,  7,  6,  9, 11, 13, 18, 15, 16, 12,
      0, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    },
    {
      14, 17, 19, 20, 21, 23, 25, 24, 22, 26, 27, 29, 31, 28, 32, 30,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    }
  },
  {
    {
      1,  3,  4,  2,  5,  6, 10,  9,  7,  8, 11, 12, 13, 16, 17, 18,
      14,  0, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    },
    {
      15, 19, 22, 20, 21, 24, 26, 25, 23, 27, 32, 29, 28, 30, 33, 34,
      31, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    }
  },
  {
    {
      1,  2,  4,  7,  6,  3,  5,  8,  9, 10, 11, 18, 15, 13, 14, 17,
      19, 12,  0, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    },
    {
      16, 21, 24, 25, 26, 20, 22, 27, 23, 28, 30, 35, 31, 29, 34, 33,
      36, 32, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    }
  },
  {
    {
      1,  2,  5,  7,  9, 10,  6,  3,  4,  8, 11, 17, 20, 18, 15, 13,
      14, 19, 12,  0, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    },
    {
      16, 21, 24, 25, 29, 28, 27, 23, 22, 26, 31, 36, 34, 37, 33, 32,
      30, 38, 35, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    }
  },
  {
    {
      1,  3,  5,  7,  9, 14, 12,  8,  6,  2,  4, 15, 18, 21, 20, 16,
      10, 11, 22, 13,  0, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    },
    {
      17, 23, 25, 27, 29, 31, 33, 30, 26, 19, 24, 35, 37, 39, 40, 36,
      32, 28, 38, 34, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    }
  },
  {
    {
      0,  2,  4,  7, 10,  9, 16, 15, 14,  8,  3,  5, 17, 19, 22, 20,
      18, 11, 13, 21, 12,  1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    },
    {
      6, 23, 25, 29, 27, 31, 30, 34, 38, 32, 24, 26, 36, 82, 39, 37,
      83, 28, 33, 84, 35, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    }
  },
  {
    {
      2,  1,  3,  4,  5,  8,  9, 11, 13, 10,  6,  7, 12, 14, 21, 24,
      23, 18, 15, 19, 22, 16,  0, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    },
    {
      20, 17, 25, 26, 27, 28, 31, 34, 35, 30, 29, 32, 33, 36, 37, 43,
      44, 38, 39, 40, 41, 42, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    }
  },
  {
    {
      1,  5,  4,  2,  3,  6,  8,  7, 11, 12,  9, 10, 13, 14, 15, 18,
      22, 21, 17, 19, 20, 25, 16,  0, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    },
    {
      26, 28, 27, 23, 24, 29, 31, 32, 38, 34, 30, 33, 35, 37, 36, 43,
      44, 40, 41, 85, 42, 86, 39, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    }
  },
  {
    {
      1,  4,  8,  6,  5,  2,  3,  7,  9, 12, 10, 11, 13, 16, 15, 14,
      18, 20, 21, 19, 23, 22, 25, 17,  0, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    },
    {
      24, 28, 33, 29, 31, 26, 27, 34, 35, 36, 32, 30, 37, 40, 38, 42,
      45, 47, 39, 43, 46, 41, 87, 44, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    }
  },
  {
    {
      0,  3,  7, 12, 10,  6,  5,  2,  4,  8,  9, 11, 13, 17, 24, 21,
      16, 15, 19, 20, 22, 25, 26, 23, 18,  1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    },
    {
      14, 28, 34, 36, 33, 35, 31, 27, 30, 29, 32, 37, 40, 46, 38, 44,
      39, 47, 48, 49, 45, 41, 42, 50, 43, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    }
  },
  {
    {
      0,  2,  5,  9, 13, 11, 10,  8,  6,  3,  4,  7, 12, 15, 19, 26,
      23, 20, 16, 14, 18, 22, 27, 24, 25, 21,  1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    },
    {
      17, 28, 31, 36, 35, 39, 37, 34, 33, 30, 29, 32, 38, 43, 86, 87,
      45, 46, 44, 41, 40, 88, 48, 89, 47, 42, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    }
  },
  {
    {
      1,  2,  4,  7, 11, 15, 12,  9, 10,  8,  5,  3,  6, 14, 16, 23,
      24, 25, 21, 20, 13, 19, 27, 28, 26, 22, 17,  0, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    },
    {
      18, 29, 31, 33, 39, 36, 35, 38, 41, 40, 34, 30, 32, 43, 48, 53,
      90, 49, 42, 50, 37, 44, 51, 47, 52, 45, 46, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    }
  },
  {
    {
      0,  2,  3,  5,  7, 11, 15, 14, 12, 10, 13,  8,  4,  9, 17, 19,
      22, 24, 26, 21, 23, 16, 20, 29, 30, 28, 25, 18,  1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    },
    {
      6, 27, 31, 33, 34, 38, 39, 37, 40, 42, 41, 35, 32, 36, 44, 45,
      87, 48, 88, 49, 46, 89, 43, 47, 90, 50, 51, 91, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1,
      M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1
    }
  },
  {
    {
      0,  1,  2,  3,  4,  5,  7,  9, 10, 13,  8, 12, 11,  6, 14, 15,
      18, 20, 24, 31, 32, 30, 28, 26, 22, 25, 33, 37, 38, 17, 19, 21,
      23, 29, 36, 34, 45, 40, 43, 42, 44, 39, 35, 46, 48, 53, 54, 51,
      61, 62, 67, 66, 68, 65, 73, 74, 71, 69, 86, 82, 90, 88, 98, 94
    },
    {
      16, 27, 41, 47, 50, 57, 58, 60, 59, 55, 52, 63, 56, 49, 64, 70,
      72, 77, 76, 80, 84, 81, 79, 78, 75, 83, 85, 87, 91, 95, 92,104,
      89, 93, 96, 99,100,102,101, 97,105,103,106,107,108,109,110,111,
      112,113,114,115,116,117,118,119,120,121,122,123,124,125,126, M1
    }
  }
};
#endif

#if  QC_MOD_LCEC
const UInt g_auiLumaRunTr14x4[5][15]=
{
    {0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0},
    {2, 3, 4, 5, 6, 5, 6, 7, 7, 7, 7, 7, 6, 4, 2},
    {2, 2, 2, 3, 3, 3, 3, 4, 4, 4, 3, 3, 3, 2, M1},
    {2, 1, 1, 2, 2, 2, 2, 2, 2, 2, 2, 2, 1, M1, M1},
    {1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, M1, M1, M1}
};
const UInt g_auiLumaRunTr18x8[5][29]=
{
    {0, 1, 2, 2, 2, 2, 2, 4, 5, 5, 6, 6, 6, 6, 6, 6, 7, 8, 7, 8, 8, 9, 10, 10, 10, 12, 10, 9, 8},
    {2, 4, 4, 6, 6, 8, 8, 10, 11, 13, 15, 13, 14, 15, 16, 18, 18, 21, 20, 21, 22, 23, 25, 25, 26, 27, 28, 29, 27},
    {2, 3, 4, 5, 4, 5, 6, 6, 7, 8, 8, 9, 9, 10, 10, 11, 11, 12, 13, 14, 15, 16, 16, 17, 18, 19, 20, 19, 19},
    {2, 1, 2, 2, 3, 3, 4, 4, 5, 5, 5, 6, 6, 7, 7, 7, 8, 9, 9, 10, 10, 11, 11, 12, 12, 13, 13, 13, 13},
    {2, 1, 2, 2, 2, 2, 3, 3, 3, 3, 4, 4, 4, 4, 4, 5, 5, 5, 5, 5, 6, 6, 6, 6, 6, 6, 7, 7, 8}
};

#if CAVLC_COEF_LRG_BLK
const UInt *g_pLumaRunTr14x4[5] = { 
  &g_auiLumaRunTr14x4[0][0], &g_auiLumaRunTr14x4[1][0], &g_auiLumaRunTr14x4[2][0], 
  &g_auiLumaRunTr14x4[3][0], &g_auiLumaRunTr14x4[4][0]
};

const UInt *g_pLumaRunTr18x8[5] = { 
  &g_auiLumaRunTr18x8[0][0], &g_auiLumaRunTr18x8[1][0], &g_auiLumaRunTr18x8[2][0], 
  &g_auiLumaRunTr18x8[3][0], &g_auiLumaRunTr18x8[4][0]
};
#endif
#endif

#if QC_MOD_LCEC
const UInt g_auiVlcTable8x8Inter[29] = {8,0,0,5,5,5,5,5,5,5,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,3};
const UInt g_auiVlcTable8x8Intra[29] = {8,0,0,5,5,5,5,5,5,5,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,3};
#if CAVLC_COEF_LRG_BLK
const UInt g_auiVlcTable16x16Intra[29] = {8,0,0,0,0,5,5,5,5,5,5,5,5,5,5,5,5,5,1,1,1,1,1,1,1,1,1,1,2};
const UInt g_auiVlcTable16x16Inter[29] = {8,0,1,1,1,1,2,2,2,2,2,2,2,6,6,6,6,6,6,3,3,3,3,3,3,3,3,3,3};
#endif
#else
const UInt g_auiVlcTable8x8[28] = {8,0,0,5,5,5,5,5,5,5,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6};
#endif

#if LCEC_INTRA_MODE
#if MTK_DCM_MPM
const UInt huff17_2[2][17]=
{
  {1, 7, 6, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 3, 2, 1, 0},
  {1, 0, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 0},
};
const UInt lengthHuff17_2[2][17]=
{
  {1, 4, 4, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 6, 6, 6, 6},
  {1, 4, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 0},
};
const UInt huff34_2[2][34]=
{
  {1, 0, 7, 6, 5, 4, 3, 2, 23, 22, 21, 20, 19, 18, 17, 16, 61, 60, 59, 58, 57, 56, 55, 54, 53, 52, 51, 50, 49, 48, 127, 126, 125, 124 },
  {1, 0, 5, 4, 3, 2, 24, 23, 22, 21, 20, 19, 18, 17, 16, 15, 14, 13, 12, 63, 62, 61, 60, 59, 58, 57, 56, 55, 54, 53, 52, 51,50, 0 }
};
const UInt lengthHuff34_2[2][34]=
{
  {1, 4,  5,  5,  5,  5,  5,  5,  6,  6,  6,  6,  6,  6,  6,  6,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  8,  8,  8, 8 },
  {1, 4,  5,  5,  5,  5,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7, 0 }
};
#else
const UInt huff17_2[2][17]=
{
  {1, 7, 6, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 3, 2, 1, 0},
  {3, 5, 9, 8, 7, 6, 5, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0}
};
const UInt lengthHuff17_2[2][17]=
{
  {1, 4, 4, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 6, 6, 6, 6},
  {2, 3, 4, 4, 4, 4, 4, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5}
};
const UInt huff34_2[2][34]=
{
  {1, 7, 13, 12, 11, 10, 9, 17, 16, 15, 14, 13, 12, 11, 10, 9, 8, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 1, 0},
  {3, 11, 10, 9, 17, 16, 15, 14, 13, 12, 11, 21, 20, 19, 18, 17, 16, 15, 14, 13, 12, 11, 10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 1, 0}
};
const UInt lengthHuff34_2[2][34]=
{
  {1, 4, 5, 5, 5, 5, 5, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 8, 8},
  {2, 4, 4, 4, 5, 5, 5, 5, 5, 5, 5, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 7, 7}
};
#endif
#endif

#if RUNLEVEL_TABLE_CUT
const UInt g_acstructLumaRun8x8[28][29] =
{
  {  1,  0, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1},
  {  2,  1,  0, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1},
  {  3,  0,  2,  1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1},
  {  4,  1,  0,  2,  3, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1},
  {  5,  0,  3,  2,  4,  1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1},
  {  6,  0,  1,  4,  5,  3,  2, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1},
  {  7,  1,  0,  2,  4,  3,  6,  5, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1},
  {  8,  0,  3,  4,  2,  1,  5,  7,  6, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1},
  {  9,  0,  5,  1,  6,  4,  8,  3,  2,  7, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1},
  { 10,  0,  1,  6,  7,  2,  9,  5,  4,  3,  8, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1},
  { 11,  1,  0,  2,  3,  6,  7,  4,  5,  8, 10,  9, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1},
  { 12,  0,  3,  2,  1,  4,  6,  5,  7, 11,  8,  9, 10, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1},
  { 13,  0,  1,  5,  6,  4,  2,  3,  7,  8, 12, 10, 11,  9, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1},
  { 14,  0,  1,  7,  2,  8,  6,  5,  3,  4, 13, 12,  9, 11, 10, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1},
  { 15,  0,  1,  2,  8,  3,  9,  7,  6,  4,  5, 14, 10, 13, 12, 11, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1},
  { 16,  0,  1,  2,  3,  4,  8,  7,  5,  9,  6, 10, 15, 11, 13, 14, 12, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1},
  { 17,  0,  3,  1,  2,  4,  5,  8,  9,  7,  6, 10, 11, 12, 16, 13, 14, 15, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1},
  { 18,  0,  1,  5,  2,  6,  4,  3,  7,  8,  9, 10, 17, 13, 14, 12, 15, 11, 16, M1, M1, M1, M1, M1, M1, M1, M1, M1, M1},
  { 19,  0,  1,  7,  8,  2,  6,  3,  9,  4,  5, 10, 18, 15, 16, 14, 11, 13, 17, 12, M1, M1, M1, M1, M1, M1, M1, M1, M1},
  { 20,  0,  9,  1, 10,  2,  8,  3,  7,  4, 16, 17,  6, 19,  5, 11, 15, 12, 14, 13, 18, M1, M1, M1, M1, M1, M1, M1, M1},
  { 21,  0,  1, 10,  2, 11,  3,  9,  5,  4, 17, 20, 18,  8,  7,  6, 12, 16, 13, 15, 19, 14, M1, M1, M1, M1, M1, M1, M1},
  { 22,  1,  0,  2,  3,  4, 10, 11,  5,  6,  9,  7, 12,  8, 13, 18, 21, 17, 19, 14, 20, 16, 15, M1, M1, M1, M1, M1, M1},
  { 23,  0,  3,  4,  2,  1,  5,  7,  6, 10, 11,  8,  9, 12, 13, 14, 22, 18, 15, 19, 20, 17, 16, 21, M1, M1, M1, M1, M1},
  { 24,  0,  5,  6,  1,  4,  3,  7,  2,  8, 10, 11,  9, 12, 15, 14, 13, 23, 16, 19, 17, 18, 21, 20, 22, M1, M1, M1, M1},
  { 25,  0,  7,  1,  8,  6,  5,  2,  9, 10,  4, 11,  3, 12, 17, 16, 13, 24, 18, 19, 15, 20, 23, 14, 21, 22, M1, M1, M1},
  { 26,  0,  1,  9, 10,  2,  8, 11,  7,  3,  6,  5, 12,  4, 19, 13, 18, 20, 14, 17, 25, 21, 16, 23, 24, 15, 22, M1, M1},
  { 27,  0,  1, 11,  2, 10, 12,  3,  9,  7,  8,  4,  6, 20, 13,  5, 14, 26, 21, 19, 18, 25, 15, 16, 17, 24, 22, 23, M1},
  { 28,  0,  1,  2, 12,  3,  4, 11, 13,  9,  5,  8, 10,  7,  6, 21, 14, 27, 15, 22, 19, 16, 20, 17, 26, 18, 25, 23, 24}
};
#else
const LastCoeffStruct g_acstructLumaRun8x8[29][127] =
{
  {
    {0,1},{0,0},{1,0},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1}
  },
  
  {
    {0,2},{0,1},{0,0},{1,1},{1,0},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1}
  },
  
  {
    {0,3},{0,0},{0,2},{0,1},{1,0},{1,2},{1,1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1}
  },
  
  {
    {0,4},{0,1},{0,0},{0,2},{0,3},{1,1},{1,0},{1,2},{1,3},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1}
  },
  
  {
    {0,5},{0,0},{0,3},{0,2},{0,4},{0,1},{1,0},{1,3},{1,2},{1,4},{1,1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1}
  },
  
  {
    {0,6},{0,0},{0,1},{0,4},{0,5},{0,3},{0,2},{1,0},{1,1},{1,4},{1,3},{1,2},{1,5},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1}
  },
  
  {
    {0,7},{0,1},{0,0},{0,2},{0,4},{0,3},{0,6},{0,5},{1,1},{1,0},{1,2},{1,4},{1,3},{1,5},{1,6},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1}
  },
  
  {
    {0,8},{0,0},{0,3},{0,4},{0,2},{0,1},{0,5},{0,7},{0,6},{1,0},{1,3},{1,4},{1,2},{1,1},{1,5},{1,6},{1,7},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1}
  },
  
  {
    {0,9},{0,0},{0,5},{0,1},{0,6},{0,4},{0,8},{0,3},{0,2},{0,7},{1,0},{1,5},{1,1},{1,6},{1,4},{1,3},{1,2},{1,8},{1,7},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1}
  },
  
  {
    {0,10},{0,0},{0,1},{0,6},{0,7},{0,2},{0,9},{0,5},{1,0},{0,4},{0,3},{0,8},{1,1},{1,6},{1,7},{1,2},{1,5},{1,3},{1,4},{1,9},{1,8},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1}
  },
  
  {
    {0,11},{0,1},{0,0},{0,2},{0,3},{0,6},{0,7},{0,4},{0,5},{0,8},{0,10},{0,9},{1,1},{1,0},{1,2},{1,3},{1,6},{1,4},{1,5},{1,7},{1,8},{1,10},
    {1,9},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1}
  },
  
  {
    {0,12},{0,0},{0,3},{0,2},{0,1},{0,4},{0,6},{0,5},{0,7},{0,11},{0,8},{0,9},{0,10},{1,0},{1,3},{1,2},{1,4},{1,1},{1,6},{1,5},{1,7},{1,11},
    {1,8},{1,9},{1,10},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1}
  },
  
  {
    {0,13},{0,0},{0,1},{0,5},{0,6},{0,4},{0,2},{0,3},{0,7},{0,8},{0,12},{0,10},{0,11},{0,9},{1,0},{1,5},{1,1},{1,6},{1,2},{1,4},{1,7},{1,3},
    {1,8},{1,10},{1,11},{1,12},{1,9},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1}
  },
  
  {
    {0,14},{0,0},{0,1},{0,7},{0,2},{0,8},{0,6},{0,5},{0,3},{0,4},{0,13},{0,12},{0,9},{0,11},{0,10},{1,0},{1,1},{1,7},{1,8},{1,2},{1,6},{1,3},
    {1,5},{1,4},{1,12},{1,13},{1,9},{1,11},{1,10},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1}
  },
  
  {
    {0,15},{0,0},{0,1},{0,2},{0,8},{0,3},{0,9},{0,7},{0,6},{0,4},{0,5},{1,0},{0,14},{0,10},{0,13},{0,12},{0,11},{1,1},{1,2},{1,8},{1,3},{1,9},
    {1,4},{1,6},{1,5},{1,13},{1,7},{1,10},{1,14},{1,12},{1,11},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1}
  },
  
  {
    {0,16},{0,0},{0,1},{0,2},{0,3},{0,4},{0,8},{0,7},{0,5},{0,9},{0,6},{0,10},{0,15},{0,11},{1,0},{0,13},{0,14},{1,1},{0,12},{1,2},{1,3},{1,4},
    {1,8},{1,5},{1,7},{1,6},{1,9},{1,10},{1,13},{1,11},{1,15},{1,12},{1,14},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1}
  },
  
  {
    {0,17},{0,0},{0,3},{0,1},{0,2},{0,4},{0,5},{0,8},{0,9},{0,7},{0,6},{0,10},{0,11},{0,12},{0,16},{1,0},{0,13},{0,14},{0,15},{1,1},{1,3},
    {1,4},{1,2},{1,8},{1,5},{1,7},{1,6},{1,9},{1,12},{1,11},{1,13},{1,16},{1,10},{1,14},{1,15},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1}
  },
  
  {
    {0,18},{0,0},{0,1},{0,5},{0,2},{0,6},{0,4},{0,3},{0,7},{0,8},{0,9},{0,10},{0,17},{0,13},{0,14},{0,12},{1,0},{0,15},{0,11},{0,16},{1,5},
    {1,1},{1,6},{1,8},{1,2},{1,3},{1,4},{1,7},{1,9},{1,13},{1,10},{1,12},{1,17},{1,15},{1,14},{1,11},{1,16},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1}
  },
  
  {
    {0,19},{0,0},{0,1},{0,7},{0,8},{0,2},{0,6},{0,3},{0,9},{0,4},{0,5},{0,10},{0,18},{0,15},{0,16},{0,14},{1,0},{0,11},{0,13},{0,17},{0,12},
    {1,1},{1,8},{1,7},{1,2},{1,3},{1,9},{1,6},{1,5},{1,4},{1,16},{1,10},{1,15},{1,14},{1,12},{1,18},{1,11},{1,13},{1,17},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1}
  },
  
  {
    {0,20},{0,0},{0,9},{0,1},{0,10},{0,2},{0,8},{0,3},{0,7},{0,4},{0,16},{0,17},{0,6},{0,19},{0,5},{0,11},{0,15},{1,0},{0,12},{1,9},{0,14},
    {0,13},{0,18},{1,1},{1,10},{1,2},{1,8},{1,3},{1,17},{1,4},{1,7},{1,5},{1,16},{1,6},{1,19},{1,11},{1,15},{1,12},{1,18},{1,13},{1,14},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1}
  },
  
  {
    {0,0},{0,21},{0,1},{0,10},{0,2},{0,11},{1,0},{0,3},{0,9},{0,5},{0,4},{0,17},{0,20},{0,18},{0,8},{0,7},{0,6},{0,12},{0,16},{0,13},{0,15},
    {0,19},{0,14},{1,1},{1,10},{1,2},{1,11},{1,4},{1,17},{1,3},{1,6},{1,5},{1,9},{1,18},{1,7},{1,20},{1,12},{1,15},{1,8},{1,14},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{1,13},{1,16},{1,19},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1}
  },
  
  {
    {0,22},{0,1},{0,0},{0,2},{0,3},{0,4},{0,10},{0,11},{0,5},{0,6},{0,9},{0,7},{0,12},{0,8},{0,13},{0,18},{0,21},{1,1},{0,17},{0,19},{1,0},
    {0,14},{0,20},{0,16},{0,15},{1,2},{1,3},{1,4},{1,5},{1,10},{1,9},{1,6},{1,11},{1,12},{1,7},{1,8},{1,13},{1,14},{1,17},{1,18},{1,19},{1,20},
    {1,21},{1,15},{1,16},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1}
  },
  
  {
    {0,23},{0,0},{0,3},{0,4},{0,2},{0,1},{0,5},{0,7},{0,6},{0,10},{0,11},{0,8},{0,9},{0,12},{0,13},{0,14},{0,22},{0,18},{0,15},{0,19},{0,20},
    {0,17},{0,16},{1,3},{1,4},{0,21},{1,0},{1,2},{1,1},{1,5},{1,10},{1,6},{1,7},{1,11},{1,9},{1,12},{1,14},{1,13},{1,8},{1,22},{1,17},{1,18},
    {1,20},{1,15},{1,16},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{1,19},{1,21},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1}
  },
  
  {
    {0,24},{0,0},{0,5},{0,6},{0,1},{0,4},{0,3},{0,7},{0,2},{0,8},{0,10},{0,11},{0,9},{0,12},{0,15},{0,14},{0,13},{0,23},{0,16},{0,19},{0,17},
    {0,18},{0,21},{0,20},{1,0},{0,22},{1,5},{1,6},{1,1},{1,3},{1,11},{1,4},{1,10},{1,2},{1,7},{1,8},{1,9},{1,12},{1,14},{1,18},{1,13},{1,21},
    {1,15},{1,19},{1,23},{1,16},{1,20},{1,17},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{1,22},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1}
  },
  {
    {0,0},{0,25},{0,7},{0,1},{0,8},{0,6},{0,5},{0,2},{0,9},{0,10},{0,4},{0,11},{0,3},{0,12},{1,0},{0,17},{0,16},{0,13},{0,24},{0,18},{0,19},
    {0,15},{0,20},{0,23},{0,14},{0,21},{0,22},{1,7},{1,1},{1,9},{1,8},{1,6},{1,10},{1,4},{1,2},{1,5},{1,3},{1,11},{1,14},{1,16},{1,12},{1,21},
    {1,22},{1,24},{1,15},{1,20},{1,13},{1,17},{1,18},{1,19},{1,23},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1}
  },
  {
    {0,0},{0,26},{0,1},{0,9},{0,10},{0,2},{0,8},{0,11},{0,7},{0,3},{0,6},{0,5},{0,12},{0,4},{0,19},{0,13},{0,18},{1,0},{0,20},{0,14},{0,17},
    {0,25},{0,21},{0,16},{0,23},{0,24},{0,15},{0,22},{1,1},{1,10},{1,9},{1,2},{1,11},{1,8},{1,7},{1,4},{1,3},{1,6},{1,12},{1,5},{1,20},{1,19},
    {1,25},{1,13},{1,18},{1,16},{1,17},{1,24},{1,22},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{1,14},{1,15},{1,21},{1,23},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1}
  },
  {
    {0,27},{0,0},{0,1},{0,11},{0,2},{0,10},{0,12},{0,3},{0,9},{0,7},{0,8},{0,4},{0,6},{0,20},{0,13},{0,5},{0,14},{0,26},{1,0},{0,21},{0,19},
    {0,18},{0,25},{0,15},{0,16},{0,17},{0,24},{0,22},{0,23},{1,1},{1,11},{1,2},{1,12},{1,3},{1,10},{1,6},{1,5},{1,20},{1,7},{1,4},{1,9},{1,8},
    {1,18},{1,13},{1,21},{1,25},{1,26},{1,23},{1,14},{1,17},{1,19},{1,22},{1,24},{1,15},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{1,16},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1}
  },
  
  {
    {0,0},{0,28},{0,1},{0,2},{0,12},{0,3},{1,0},{0,4},{0,11},{0,13},{0,9},{0,5},{0,8},{0,10},{0,7},{0,6},{0,21},{0,14},{0,27},{0,15},{0,22},
    {0,19},{0,16},{0,20},{0,17},{0,26},{0,18},{1,1},{0,25},{0,23},{0,24},{1,2},{1,12},{1,3},{1,4},{1,11},{1,13},{1,7},{1,5},{1,6},{1,8},{1,10},
    {1,9},{1,22},{1,14},{1,15},{1,20},{1,23},{1,17},{1,19},{1,25},{1,26},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{1,16},{1,18},{1,21},{1,24},{1,27},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},
    {M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1},{M1,M1}
  },
  
  {
    {0,0},{0,1},{0,2},{0,3},{0,4},{0,5},{0,13},{0,6},{0,10},{0,7},{0,8},{0,12},{0,11},{0,9},{0,14},{0,15},{1,0},{0,29},{0,16},{0,30},{0,17},
    {0,31},{0,24},{0,32},{0,18},{0,25},{0,23},{1,1},{0,22},{0,33},{0,21},{0,19},{0,20},{0,26},{0,35},{0,42},{0,34},{0,27},{0,28},{0,41},{0,37},
    {1,2},{0,39},{0,38},{0,40},{0,36},{0,43},{1,3},{0,44},{1,13},{1,4},{0,47},{1,10},{0,45},{0,46},{1,9},{1,12},{1,5},{1,6},{1,8},{1,7},{0,48},
    {0,49},{1,11},{1,14},{0,53},{0,51},{0,50},{0,52},{0,57},{1,15},{0,56},{1,16},{0,54},{0,55},{1,24},{1,18},{1,17},{1,23},{1,22},{1,19},
    {1,21},{0,59},{1,25},{1,20},{1,26},{0,58},{1,27},{0,61},{1,32},{0,60},{1,28},{1,30},{1,33},{0,63},{1,29},{1,34},{1,39},{0,62},{1,35},
    {1,36},{1,38},{1,37},{1,41},{1,31},{1,40},{1,42},{1,43},{1,44},{1,45},{1,46},{1,47},{1,48},{1,49},{1,50},{1,51},{1,52},{1,53},{1,54},
    {1,55},{1,56},{1,57},{1,58},{1,59},{1,60},{1,61},{1,62}
  }
};
#endif

// Mode-Dependent DCT/DST 
#if INTRA_DST_TYPE_7
const short g_as_DST_MAT_4 [4][4]=
{
  {29,   55,    74,   84},
  {74,   74,    0 ,  -74},
  {84,  -29,   -74,   55},
  {55,  -84,    74,  -29},
};
// Mapping each Unified Directional Intra prediction direction to DCT/DST transform 
// 0 implies use DCT, 1 implies DST
#if ADD_PLANAR_MODE && !HHI_DMM_INTRA
const UChar g_aucDCTDSTMode_Vert[NUM_INTRA_MODE] =
#elif !ADD_PLANAR_MODE && HHI_DMM_INTRA
const UChar g_aucDCTDSTMode_Vert[34+NUM_DMM_INTRA] =
#elif ADD_PLANAR_MODE && HHI_DMM_INTRA
const UChar g_aucDCTDSTMode_Vert[NUM_INTRA_MODE+NUM_DMM_INTRA] =
#else
const UChar g_aucDCTDSTMode_Vert[34] =
#endif
{ //0  1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 16 17 18 19 20 21 22 23 24 25 26 27 28 29 30 31 32 33
  1, 0, 0, 1, 1, 1, 1, 1, 0, 0, 1, 1, 1, 1, 1, 1, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0 ,0 
#if ADD_PLANAR_MODE
,1
#endif
#if HHI_DMM_INTRA
,0,0,0,0,0,0,0,0
#endif
};
#if ADD_PLANAR_MODE && !HHI_DMM_INTRA
const UChar g_aucDCTDSTMode_Hor[NUM_INTRA_MODE] =
#elif !ADD_PLANAR_MODE && HHI_DMM_INTRA
const UChar g_aucDCTDSTMode_Hor[34+NUM_DMM_INTRA] =
#elif ADD_PLANAR_MODE && HHI_DMM_INTRA
const UChar g_aucDCTDSTMode_Hor[NUM_INTRA_MODE+NUM_DMM_INTRA] =
#else
const UChar g_aucDCTDSTMode_Hor[34] =
#endif
{ //0  1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 16 17 18 19 20 21 22 23 24 25 26 27 28 29 30 31 32 33
  0, 1, 0, 1, 1, 0, 0, 1, 1, 1, 1, 1, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1 ,1 
#if ADD_PLANAR_MODE
,1
#endif
#if HHI_DMM_INTRA
,0,0,0,0,0,0,0,0
#endif
};
#endif  // for INTRA_DST_TYPE_7



// ====================================================================================================================
// ADI
// ====================================================================================================================

#if FAST_UDI_USE_MPM
const UChar g_aucIntraModeNumFast[7] =
{
  3,  //   2x2
  8,  //   4x4
  8,  //   8x8
  3,  //  16x16   
  3,  //  32x32   
  3,  //  64x64   
  3   // 128x128  
};
#else // FAST_UDI_USE_MPM
#if SAMSUNG_FAST_UDI_MODESET==0
const UChar g_aucIntraModeNumFast[7] =
{
  3,  //   2x2
  9,  //   4x4
  9,  //   8x8
  4,  //  16x16   33
  4,  //  32x32   33
  5,  //  64x64   33
  4   // 128x128  33
};
#else
const UChar g_aucIntraModeNumFast[7] =
{
  3,  //   2x2
  9,  //   4x4
  9,  //   8x8
  9,  //  16x16   33
  9,  //  32x32   33
  5,  //  64x64   33
  4   // 128x128  33
};
#endif
#endif // FAST_UDI_USE_MPM

// chroma

const UChar g_aucConvertTxtTypeToIdx[4] = { 0, 1, 1, 2 };

// ====================================================================================================================
// Angular Intra prediction
// ====================================================================================================================

// g_aucAngIntraModeOrder
//   Indexing this array with the mode indicated in the bitstream
//   gives a logical index used in the prediction functions.
#if ADD_PLANAR_MODE
const UChar g_aucAngIntraModeOrder[NUM_INTRA_MODE] =
#else
const UChar g_aucAngIntraModeOrder[34] =
#endif
{     //  ModeOrder LogicalOrderInPredFunctions
  9,  //  0 VER     DC
  25, //  1 HOR     VER-8 (diagonal from top-left to bottom-right = HOR-8)
  0,  //  2 DC      VER-7
  1,  //  3 VER-8   VER-6
  5,  //  4 VER-4   VER-5
  13, //  5 VER+4   VER-4
  17, //  6 VER+8   VER-3
  21, //  7 HOR-4   VER-2
  29, //  8 HOR+4   VER-1
  33, //  9 HOR+8   VER
  3,  // 10 VER-6   VER+1
  7,  // 11 VER-2   VER+2
  11, // 12 VER+2   VER+3
  15, // 13 VER+6   VER+4
  19, // 14 HOR-6   VER+5
  23, // 15 HOR-2   VER+6
  27, // 16 HOR+2   VER+7
  31, // 17 HOR+6   VER+8
  2,  // 18 VER-7   HOR-7
  4,  // 19 VER-5   HOR-6
  6,  // 20 VER-3   HOR-5
  8,  // 21 VER-1   HOR-4
  10, // 22 VER+1   HOR-3
  12, // 23 VER+3   HOR-2
  14, // 24 VER+5   HOR-1
  16, // 25 VER+7   HOR
  18, // 26 HOR-7   HOR+1
  20, // 27 HOR-5   HOR+2
  22, // 28 HOR-3   HOR+3
  24, // 29 HOR-1   HOR+4
  26, // 30 HOR+1   HOR+5
  28, // 31 HOR+3   HOR+6
  30, // 32 HOR+5   HOR+7
  32, // 33 HOR+7   HOR+8
#if ADD_PLANAR_MODE
  0, // PLANAR_IDX PLANAR PLANAR (add any other mode earlier; this mode is designed to be the last in the list)
#endif
};

const UChar g_aucIntraModeNumAng[7] =
{
  3,  //   2x2
  17,  //   4x4
  34,  //   8x8
  34,  //  16x16
  34,  //  32x32
  3,  //  64x64
  5   // 128x128
};

const UChar g_aucIntraModeBitsAng[7] =
{
  2,  //   2x2     3   1+1
  5,  //   4x4    17   4+1
  6,  //   8x8    34   5+esc
  6,  //  16x16   34   5+esc
  6,  //  32x32   34   5+esc
  2,  //  64x64    3   1+1
  3   // 128x128   5   2+1
};

const UChar g_aucAngModeMapping[4][34] = // intra mode conversion for most probable
{
  {2,3,2,2,4, 4,4,0,0,0, 0,0,0,0,2, 2,2,2,2,2, 2,1,1,1,1, 1,1,1,1,1, 2,2,2,2},               // conversion to 5 modes
  {2,3,3,2,4, 4,4,2,0,0, 0,2,5,5,5, 2,6,6,3,2, 7,7,7,2,1, 1,1,2,8,8, 8,2,2,2},               // conversion to 9 modes
  {2,3,3,10,10, 4,11,11,0,0, 0,12,12,5,5, 13,13,6,14,14, 7,7,15,15,1, 1,1,16,16,8, 8,2,2,9}, // conversion to 17 modes
  {2,2,2,2,2, 2,2,0,0,0, 0,0,0,0,2, 2,2,2,2,2, 2,1,1,1,1, 1,1,1,1,1, 2,2,2,2}                // conversion to 3 modes
};

#if HHI_DMM_INTRA
const WedgeResolution g_aeWedgeResolutionList[5] = 
{
  HALF_PEL,    //  4x4
  HALF_PEL,    //  8x8
  FULL_PEL,    // 16x16
  DOUBLE_PEL,  // 32x32
  DOUBLE_PEL   // 64x64
};

const UChar g_aucWedgeFullBitsListIdx[7] =
{
  0,   //   2x2
  7,   //   4x4    WedgeListSize[  HALF_PEL]   86
  10,  //   8x8    WedgeListSize[  HALF_PEL]  782
  11,  //  16x16   WedgeListSize[  FULL_PEL] 1394
  11,  //  32x32   WedgeListSize[DOUBLE_PEL] 1503
  13,  //  64x64   WedgeListSize[DOUBLE_PEL] 6079
  0    // 128x128  
};

const UChar g_aucIntraSizeIdxToWedgeSize[7] =
{
  2,
  4,
  8,
  16,
  32,
  64,
  128  
};

const UChar g_aucAdditionalIntraModeList[NUM_DMM_INTRA] =
{
  DMM_WEDGE_FULL_IDX,
  DMM_WEDGE_FULL_D_IDX,
  DMM_WEDGE_PREDTEX_IDX,
  DMM_WEDGE_PREDTEX_D_IDX,
  DMM_CONTOUR_PREDTEX_IDX,
  DMM_CONTOUR_PREDTEX_D_IDX,
  DMM_WEDGE_PREDDIR_IDX,
  DMM_WEDGE_PREDDIR_D_IDX
};

double g_dDeltaDCsQuantOffset = 0.0;
#endif

// ====================================================================================================================
// Bit-depth
// ====================================================================================================================

UInt g_uiBitDepth     = 8;    // base bit-depth
UInt g_uiBitIncrement = 0;    // increments
UInt g_uiIBDI_MAX     = 255;  // max. value after  IBDI
UInt g_uiBASE_MAX     = 255;  // max. value before IBDI

// ====================================================================================================================
// Misc.
// ====================================================================================================================
#if SB_DEBUG
Bool g_bEncoding = false ;
#endif
Char  g_aucConvertToBit  [ MAX_CU_SIZE+1 ];

#if ENC_DEC_TRACE
FILE*  g_hTrace = NULL;
const Bool g_bEncDecTraceEnable  = true;
const Bool g_bEncDecTraceDisable = false;
Bool   g_bJustDoIt = false;
UInt64 g_nSymbolCounter = 0;
#endif
// ====================================================================================================================
// Scanning order & context model mapping
// ====================================================================================================================

// scanning order table
UInt* g_auiFrameScanXY[ MAX_CU_DEPTH  ];
UInt* g_auiFrameScanX [ MAX_CU_DEPTH  ];
UInt* g_auiFrameScanY [ MAX_CU_DEPTH  ];
#if QC_MDCS
UInt* g_auiSigLastScan[3][ MAX_CU_DEPTH ];
#endif //QC_MDCS

#if PCP_SIGMAP_SIMPLE_LAST
UInt g_uiCtxXYOffset[ MAX_CU_DEPTH ] =
{
  15, 15, 15, 8, 3, 0, 0
};

UInt g_uiCtxXY[ 31 ] =
{
  0, 1, 2, 3, 3, 4, 4, 5, 5, 5, 5, 6, 6, 6, 6, 7, 7, 7, 7, 8, 8, 8, 8, 9, 9, 9, 9, 10, 10, 10, 10
};
#endif

// scanning order to 8x8 context model mapping table
UInt  g_auiAntiScan8  [64];

#if E253
// Rice parameters for absolute transform levels
const UInt g_auiGoRiceRange[4] =
{
  7, 20, 42, 70
};

const UInt g_auiGoRicePrefixLen[4] =
{
  8, 10, 11, 8
};

const UInt g_aauiGoRiceUpdate[4][16] =
{
  {
    0, 0, 1, 1, 2, 2, 2, 2, 2, 2, 2, 2, 2, 3, 3, 3
  },
  {
    1, 1, 1, 1, 2, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, 3
  },
  {
    2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, 3, 3
  },
  {
    3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3
  }
};
#endif

// initialize g_auiFrameScanXY
Void initFrameScanXY( UInt* pBuff, UInt* pBuffX, UInt* pBuffY, Int iWidth, Int iHeight )
{
  Int x, y, c = 0;
  
  // starting point
  pBuffX[ c ] = 0;
  pBuffY[ c ] = 0;
  pBuff[ c++ ] = 0;
  
  // loop
  x=1; y=0;
  while (1)
  {
    // decrease loop
    while ( x>=0 )
    {
      if ( x >= 0 && x < iWidth && y >= 0 && y < iHeight )
      {
        pBuffX[ c ] = x;
        pBuffY[ c ] = y;
        pBuff[ c++ ] = x+y*iWidth;
      }
      x--; y++;
    }
    x=0;
    
    // increase loop
    while ( y>=0 )
    {
      if ( x >= 0 && x < iWidth && y >= 0 && y < iHeight )
      {
        pBuffX[ c ] = x;
        pBuffY[ c ] = y;
        pBuff[ c++ ] = x+y*iWidth;
      }
      x++; y--;
    }
    y=0;
    
    // termination condition
    if ( c >= iWidth*iHeight ) break;
  }
  
  // LTR_2D_CONTEXT_MAPPING
  if (iWidth == 8 && iHeight == 8)
  {
    for( c = 0; c < iWidth*iHeight; c++)
    {
      g_auiAntiScan8[pBuff[c]] = c;
    }
  }
}

#if QC_MDCS
Void initSigLastScan(UInt* pBuffZ, UInt* pBuffH, UInt* pBuffV, Int iWidth, Int iHeight, Int iDepth)
{

    memcpy(pBuffZ, g_auiFrameScanXY[iDepth], sizeof(UInt)*iWidth*iHeight);

    UInt uiCnt = 0;
    for(Int iY=0; iY < iHeight; iY++)
    {
      for(Int iX=0; iX < iWidth; iX++)
      {
        pBuffH[uiCnt] = iY*iWidth + iX;
        uiCnt ++;
      }
    }

    uiCnt = 0;
    for(Int iX=0; iX < iWidth; iX++)
    {
      for(Int iY=0; iY < iHeight; iY++)
      {
        pBuffV[uiCnt] = iY*iWidth + iX;
        uiCnt ++;
      }
    }
    
}
#endif //QC_MDCS

#if CHROMA_CODEWORD_SWITCH 
const UChar ChromaMapping[2][5] = 
{
  {0, 1, 3, 2, 4},
  {0, 1, 2, 4, 3}
};
#endif

#if HHI_DMM_INTRA
std::vector<std::vector<TComWedgelet> > g_aacWedgeLists;
std::vector<std::vector<TComWedgeRef> > g_aacWedgeRefLists;

Void initWedgeLists()
{
  for( UInt ui = g_aucConvertToBit[DMM_WEDGEMODEL_MIN_SIZE]; ui < (g_aucConvertToBit[DMM_WEDGEMODEL_MAX_SIZE]+1); ui++ )
  {
    UInt uiWedgeBlockSize = ((UInt)DMM_WEDGEMODEL_MIN_SIZE)<<ui;
    std::vector<TComWedgelet> acWedgeList;
    std::vector<TComWedgeRef> acWedgeRefList;
    createWedgeList( uiWedgeBlockSize, uiWedgeBlockSize, acWedgeList, acWedgeRefList, g_aeWedgeResolutionList[ui] );
    g_aacWedgeLists.push_back( acWedgeList );
    g_aacWedgeRefLists.push_back( acWedgeRefList );
  }

  return;
}

Void createWedgeList( UInt uiWidth, UInt uiHeight, std::vector<TComWedgelet> &racWedgeList, std::vector<TComWedgeRef> &racWedgeRefList, WedgeResolution eWedgeRes )
{
  assert( uiWidth == uiHeight );

  UChar uhStartX, uhStartY, uhEndX, uhEndY;
  Int    iStepStartX, iStepStartY, iStepEndX, iStepEndY;

  UInt uiBlockSize = 0;
  switch( eWedgeRes )
  {
  case( DOUBLE_PEL ): { uiBlockSize = (uiWidth>>1); break; }
  case(   FULL_PEL ): { uiBlockSize =  uiWidth;     break; }
  case(   HALF_PEL ): { uiBlockSize = (uiWidth<<1); break; }
  }

  Bool bValid;
  TComWedgelet cTempWedgelet( uiWidth, uiHeight );
  TComWedgeRef cTempWedgeRef;
  for( UInt uiOri = 0; uiOri < 6; uiOri++ )
  {
    // init the edge line parameters for each of the 6 wedgelet types
    switch( uiOri )
    {
    case( 0 ): {  uhStartX = 0;               uhStartY = 0;               uhEndX = 0;               uhEndY = 0;               iStepStartX = +1; iStepStartY =  0; iStepEndX =  0; iStepEndY = +1; break; }
    case( 1 ): {  uhStartX = (uiBlockSize-1); uhStartY = 0;               uhEndX = (uiBlockSize-1); uhEndY = 0;               iStepStartX =  0; iStepStartY = +1; iStepEndX = -1; iStepEndY =  0; break; }
    case( 2 ): {  uhStartX = (uiBlockSize-1); uhStartY = (uiBlockSize-1); uhEndX = (uiBlockSize-1); uhEndY = (uiBlockSize-1); iStepStartX = -1; iStepStartY =  0; iStepEndX =  0; iStepEndY = -1; break; }
    case( 3 ): {  uhStartX = 0;               uhStartY = (uiBlockSize-1); uhEndX = 0;               uhEndY = (uiBlockSize-1); iStepStartX =  0; iStepStartY = -1; iStepEndX = +1; iStepEndY =  0; break; }
    case( 4 ): {  uhStartX = 0;               uhStartY = 0;               uhEndX = 0;               uhEndY = (uiBlockSize-1); iStepStartX = +1; iStepStartY =  0; iStepEndX = +1; iStepEndY =  0; break; }
    case( 5 ): {  uhStartX = (uiBlockSize-1); uhStartY = 0;               uhEndX = 0;               uhEndY = 0;               iStepStartX =  0; iStepStartY = +1; iStepEndX =  0; iStepEndY = +1; break; }
    }

    for( Int iK = 0; iK < uiBlockSize; iK++ )
    {
      for( Int iL = 0; iL < uiBlockSize; iL++ )
        {
        cTempWedgelet.setWedgelet( uhStartX + (iK*iStepStartX) , uhStartY + (iK*iStepStartY), uhEndX + (iL*iStepEndX), uhEndY + (iL*iStepEndY), (UChar)uiOri, eWedgeRes );

        bValid = cTempWedgelet.checkNotPlain();
        if( bValid )
        {
          for( UInt uiPos = 0; uiPos < racWedgeList.size(); uiPos++ )
          {
           if( !cTempWedgelet.checkNotIdentical( racWedgeList[uiPos].getPattern() ) )
           {
             cTempWedgeRef.setWedgeRef( uhStartX + (iK*iStepStartX) , uhStartY + (iK*iStepStartY), uhEndX + (iL*iStepEndX), uhEndY + (iL*iStepEndY), uiPos );
             racWedgeRefList.push_back( cTempWedgeRef );
             bValid = false;
             break;
           }
          }
        }
        if( bValid )
        {
          for( UInt uiPos = 0; uiPos < racWedgeList.size(); uiPos++ )
          {
            if( !cTempWedgelet.checkNotInvIdentical( racWedgeList[uiPos].getPattern() ) )
            {
              cTempWedgeRef.setWedgeRef( uhStartX + (iK*iStepStartX) , uhStartY + (iK*iStepStartY), uhEndX + (iL*iStepEndX), uhEndY + (iL*iStepEndY), uiPos );
              racWedgeRefList.push_back( cTempWedgeRef );
              bValid = false;
              break;
            }
          }
        }
        if( bValid )
        {
          racWedgeList.push_back( cTempWedgelet );
        }
      }
    }
  }
}
#endif
