

/** \file     TComPicYuv.h
    \brief    picture YUV buffer class (header)
*/

#ifndef __TCOMPICYUV__
#define __TCOMPICYUV__

#include <stdio.h>
#include "CommonDef.h"

// ====================================================================================================================
// Class definition
// ====================================================================================================================

/// picture YUV buffer class
class TComPicYuv
{
private:
  
  // ------------------------------------------------------------------------------------------------
  //  YUV buffer
  // ------------------------------------------------------------------------------------------------
  
  Pel*  m_apiPicBufY;           ///< Buffer (including margin)
  Pel*  m_apiPicBufU;
  Pel*  m_apiPicBufV;
  
  Pel*  m_piPicOrgY;            ///< m_apiPicBufY + m_iMarginLuma*getStride() + m_iMarginLuma
  Pel*  m_piPicOrgU;
  Pel*  m_piPicOrgV;
  
  // ------------------------------------------------------------------------------------------------
  //  Parameter for general YUV buffer usage
  // ------------------------------------------------------------------------------------------------
  
  Int   m_iPicWidth;            ///< Width of picture
  Int   m_iPicHeight;           ///< Height of picture
  
  Int   m_iCuWidth;             ///< Width of Coding Unit (CU)
  Int   m_iCuHeight;            ///< Height of Coding Unit (CU)
  UInt  m_uiMaxCuDepth;         ///< maximum coding unit depth
  Int   m_iBaseUnitWidth;       ///< Width of Base Unit (with maximum depth or minimum size, m_iCuWidth >> Max. Depth)
  Int   m_iBaseUnitHeight;      ///< Height of Base Unit (with maximum depth or minimum size, m_iCuHeight >> Max. Depth)
  Int   m_iNumCuInWidth;
  
  Int   m_iLumaMarginX;
  Int   m_iLumaMarginY;
  Int   m_iChromaMarginX;
  Int   m_iChromaMarginY;
  
  Bool  m_bIsBorderExtended;
  
protected:
  Void  xExtendPicCompBorder (Pel* piTxt, Int iStride, Int iWidth, Int iHeight, Int iMarginX, Int iMarginY);
  Void  xSetPels( Pel* piPelSource , Int iSourceStride, Int iWidth, Int iHeight, Pel iVal );
  
public:
  TComPicYuv         ();
  virtual ~TComPicYuv();
  
  // ------------------------------------------------------------------------------------------------
  //  Memory management
  // ------------------------------------------------------------------------------------------------
  
  Void  create      ( Int iPicWidth, Int iPicHeight, UInt uiMaxCUWidth, UInt uiMaxCUHeight, UInt uiMaxCUDepth );
  Void  destroy     ();
  
  Void  createLuma  ( Int iPicWidth, Int iPicHeight, UInt uiMaxCUWidth, UInt uiMaxCUHeight, UInt uhMaxCUDepth );
  Void  destroyLuma ();
  
  // ------------------------------------------------------------------------------------------------
  //  Get information of picture
  // ------------------------------------------------------------------------------------------------
  
  Int   getWidth    ()     { return  m_iPicWidth;    }
  Int   getHeight   ()     { return  m_iPicHeight;   }

  UInt  getMaxCuWidth ()   { return  (UInt)m_iCuWidth;   }
  UInt  getMaxCuHeight()   { return  (UInt)m_iCuHeight;  }
  UInt  getMaxCuDepth ()   { return  m_uiMaxCuDepth;     }
  
  Int   getStride   ()     { return (m_iPicWidth     ) + (m_iLumaMarginX  <<1); }
  Int   getCStride  ()     { return (m_iPicWidth >> 1) + (m_iChromaMarginX<<1); }
  
  Int   getLumaMargin   () { return m_iLumaMarginX;  }
  Int   getChromaMargin () { return m_iChromaMarginX;}
  
  Void  getLumaMinMax( Int* pMin, Int* pMax );
  
  // ------------------------------------------------------------------------------------------------
  //  Access function for picture buffer
  // ------------------------------------------------------------------------------------------------
  
  //  Access starting position of picture buffer with margin
  Pel*  getBufY     ()     { return  m_apiPicBufY;   }
  Pel*  getBufU     ()     { return  m_apiPicBufU;   }
  Pel*  getBufV     ()     { return  m_apiPicBufV;   }
  
  //  Access starting position of original picture
  Pel*  getLumaAddr ()     { return  m_piPicOrgY;    }
  Pel*  getCbAddr   ()     { return  m_piPicOrgU;    }
  Pel*  getCrAddr   ()     { return  m_piPicOrgV;    }
  
  //  Access starting position of original picture for specific coding unit (CU) or partition unit (PU)
  Pel*  getLumaAddr ( Int iCuAddr );
  Pel*  getCbAddr   ( Int iCuAddr );
  Pel*  getCrAddr   ( Int iCuAddr );
  Pel*  getLumaAddr ( Int iCuAddr, Int uiAbsZorderIdx );
  Pel*  getCbAddr   ( Int iCuAddr, Int uiAbsZorderIdx );
  Pel*  getCrAddr   ( Int iCuAddr, Int uiAbsZorderIdx );
  
  // ------------------------------------------------------------------------------------------------
  //  Miscellaneous
  // ------------------------------------------------------------------------------------------------

  // sample to block and block to sample conversion
  Void  getTopLeftSamplePos( Int iCuAddr, Int iAbsZorderIdx, Int& riX, Int& riY );
  Void  getCUAddrAndPartIdx( Int iX, Int iY, Int& riCuAddr, Int& riAbsZorderIdx );
  
  //  Copy function to picture
  Void  copyToPic       ( TComPicYuv*  pcPicYuvDst );
  Void  copyToPicLuma   ( TComPicYuv*  pcPicYuvDst );
  Void  copyToPicCb     ( TComPicYuv*  pcPicYuvDst );
  Void  copyToPicCr     ( TComPicYuv*  pcPicYuvDst );
  
  //  Extend function of picture buffer
  Void  extendPicBorder      ();
  
  //  Dump picture
  Void  dump (char* pFileName, Bool bAdd = false);
  
  // Set border extension flag
  Void  setBorderExtension(Bool b) { m_bIsBorderExtended = b; }
#if FIXED_ROUNDING_FRAME_MEMORY
  Void  xFixedRoundingPic();
#endif  

  // Set Function 
  Void  setLumaTo    ( Pel pVal );  
  Void  setChromaTo  ( Pel pVal );  

};// END CLASS DEFINITION TComPicYuv

void calcMD5(TComPicYuv& pic, unsigned char digest[16]);

#endif // __TCOMPICYUV__

